#include <cstdlib>
#include <iostream>
/*
	Copyright Hairong Liu (lhrbss@gmail.com).
    This software can be used for research purposes only.
	This software or its derivatives must not be publicly distributed
	without a prior consent from the author (Hairong Liu).

	This is a software for densest subgraph partition.

	Usage:
	1, SetNumOfVar          : set number of variables
	2, SetMaxNumOfTerms     : set maximal number of terms. 
	3, SetRoundInteger (opt): set a large integer number. If not called, default is 100000
	internally used to round all values to integers, required by QPBO 
	4: AddTerm              : call multiple times to add all terms
	5: SetInitPermutation    (opt): give a good initial permutation to accelerate the speed. If not provided, automatically calculate one using some heursitc rules	
	6: Solve                : solve problem
	7: GetNumOfPartition    : get number of partitions
	8: GetPartition         : get partition result.	 
*/
struct IntDouble
{
	int I;
	double D;
};

struct FLenBuf
{
	int* pBuf;
	int Dim;
	int Len;
};

struct Term
{
	int* Vars;
	int NumElem;
	double c;
};

class SubPermutation
{
public:
	int StartPos, EndPos; //pos in the partition
	int NumVars;	
	double AveReward;
	bool CanPartition;
	int Singature;
	SubPermutation* pPre, *pNext;
};

class DSP
{
public:	
	DSP()
	{
		m_NumVar = 0;
		m_NumOfTerms = 0;	
		m_MaxNumOfTerms = 0;		
		m_RoundInteger = 100000;
		m_KeyNumber = 100;
		m_RelEps = 0.0001;
		m_pHead = NULL;	
		m_pTerms = NULL;
		m_pXOrder = NULL;
		m_pInvXOrder = NULL;
		m_pRewards = NULL;
		m_pVertex2Term = NULL;
		m_pIntBuf1 = NULL;
		m_pIntBuf2 = NULL;
		m_pDoubleBuf1 = NULL;
		m_pDoubleBuf2 = NULL;
		m_pIDBuf = NULL;		
		IsInitalized = false;
	}

	SubPermutation* m_pHead;	// min-partition
	Term* m_pTerms;
	int m_NumOfTerms;
	int m_MaxNumOfTerms;
	int m_NumVar;

	int* m_pXOrder, *m_pInvXOrder;
	double* m_pRewards; // index by pos	
	bool IsInitalized;

	int* m_pIntBuf1;
	int* m_pIntBuf2;
	double* m_pDoubleBuf1;
	double* m_pDoubleBuf2;
	IntDouble* m_pIDBuf;

	FLenBuf* m_pVertex2Term;

	int m_RoundInteger;	
	int m_KeyNumber;
	double m_RelEps;

	void SetRoundInteger(int ri)
	{
		m_RoundInteger = ri;
	}

	void SetNumOfVar(int numVar)
	{
		m_NumVar = numVar;	
		m_pXOrder = new int[m_NumVar];
		m_pInvXOrder = new int[m_NumVar];
		m_pRewards = new double[m_NumVar];
		m_pIntBuf1 = new int[m_NumVar];
		m_pIntBuf2 = new int[m_NumVar];
		m_pDoubleBuf1 = new double[m_NumVar];
		m_pDoubleBuf2 = new double[m_NumVar];
		m_pIDBuf  = new IntDouble[m_NumVar];
	}

	void SetMaxNumOfTerms(int MaxNumTerm)
	{
		m_MaxNumOfTerms = MaxNumTerm;	
		m_pTerms = new Term[m_MaxNumOfTerms];
	}

	void SetInitPermutation(int* pInitOrder)
	{
		for(int i=0;i<m_NumVar;i++)
		{
			int temp = pInitOrder[i];
			m_pXOrder[i] = temp;
			m_pInvXOrder[temp] = i;
		}
		
		IsInitalized = true;
	}

	void AddTerm(int* termIndices, int numVer, double c)
	{
		if(m_NumOfTerms>=m_MaxNumOfTerms)		
			return;		

		m_pTerms[m_NumOfTerms].NumElem = numVer;
		m_pTerms[m_NumOfTerms].Vars = new int[numVer];
		for(int i=0;i<numVer;i++)
			m_pTerms[m_NumOfTerms].Vars[i] = termIndices[i];
		m_pTerms[m_NumOfTerms].c = c;	
		m_NumOfTerms ++;
	}

	/*
	 Read problem from a binary file
	*/
	void ReadProblem(char* FileName);

	/*
	 Output partitions to a binary file
	*/
	void OuputPartition(char* FileName);

	/*
	 Initialization
	*/
	void Init(void);

	/*
	 Release all memories
	*/
	void Release(void);	

	/*
	 Relation between vertex and terms
	*/
	void BuildVertex2Term(void);

	/*
	 Order by averaging the weight of edge on each vertex, then sorting 
	 NewOrder is the best order, and the maximal density is returned.
	*/
	double GreedyOptimalOrder(int StartPos, int EndPos, int* pNewOrder, double* pNewReward);

	/*
	 Update variables in the range [StartPos,EndPos], using NewOrder,
	 also update rewards accordingly.
	*/
	void UpdateOrder(int StartPos, int EndPos, int* pNewOrder);

	/*
	  Update rewards for a range of variables. 
	*/
	void UpdateReward(int StartPos, int EndPos);

	/*
	  Update vertex-term relation
	*/
	void UpdateVertexTermRelation(int StartPos, int EndPos);

	/*
	  Construct a min-partition from a range of variables. 
	  Return pointers to new allocated maximal min-sub-permutations.
	*/
	int MinPartition(int StartPos, int EndPos, SubPermutation*** pppOutKTs);

	/*
	 Divide a range of variables into conditional independent components.
	 Each conditional independent component is returned as a min-sub-permutation.
	*/
	int CIP(int StartPos, int EndPos, SubPermutation*** pppOutKTs);	

	/*
	Partition function. If can partition, return true, and pOutKTs 
	output a min-partition; otherwise return false, and pOutKTs is empty.	
	It first tries to find a partition by some fast heuristic algorithm; if fail, then
	solves a submodular MRF problem.
	*/
	int ReOrderVerticesWithinMMSP(SubPermutation* pInKT,SubPermutation*** pppOutKTs);	

	/*
	  Min-merge algorithm
	  Scan the whole list, if some a<=b<=c<=...<=d type min-sub-permutation appears, merge them into one. 
	*/
	void MinMerge(void);

	/*
	The function to solve whole problem.
	*/
	void Solve(void);	

	/*
	Get the number of components
	*/
	int GetNumOfPartition(void)
	{
		int count = 0;

		SubPermutation* ptr = m_pHead;
		while(ptr!=NULL)
		{	
			count++;
			ptr = ptr->pNext;
		}

		return count;
	}

	/*
	Get the partition result. For each sub-permutation, only output its ending position
	*/
	void GetPartition(int* pEndPos, int* pMark);
};
