#include <mex.h>
#include <math.h>
#include <vector>
#include <map>
#include <set>
#include <windows.h>
using namespace std;

/*
  [Maxima,Support,FValue] = SEA(AdjMatrixElems,ExpNeighbors,StartPoint,Para);
  Input:
  AdjMatrixElems: array of (i,j,av), i<j, and av is the affinity value between vertex i and vertex j, a sparse representation of affinity matrix.
  ExpNeighbors  : for each vertex i, define the neighbors it can expand, note that in some applications, such as maximum common subgraph problem, this set is significantly smaller than 
  neighbors of vertex i, thus can speed up the algorithm. If not provided, it will be automatically computated according to affinity values.
  StartPoint    : a point in the space of simplex, array of cells, each cell contain two arrays, one array is the support of this point, the other array is the cooridnate of this point.
  Para          : parameter, the meaning of each component is described below.
  Output:
  Support: the support of a maxima
  Maxima : probabilistic cooridnate of a maxima
  FValue : density at a maxima
*/

typedef vector<double> DoubleVector;
typedef vector<DoubleVector> DoubleArray;

typedef vector<int> IntVector;
typedef vector<IntVector> IntArray;

typedef struct ProPoint_TYP{
	IntVector Support;
	DoubleVector Prob;
}ProPoint;
typedef vector<ProPoint> ProPointVector;

typedef struct IntDouble_TYP{
	int I;
	double D;
}IntDouble;
typedef vector<IntDouble> IDVector;
typedef vector<IDVector>  IDArray;

typedef struct IntIntDouble_TYP{
	int I[2];
	double D;
}IntIntDouble;
typedef vector<IntIntDouble> IIDVector;

//seem many parameters, but in fact only the first need to be set, others can be constant.
typedef struct Parameter_TYP{	
	int NumOfVertices; // Number of Vertices in the graph, must provided, since the affinity matrix is in sparse form
	int K;             // Control the number of added vertices in expansion phase
	double MinIncrease;// Smallest increase, the aim is to speed up algorithm, for simplicity, you can provide a small value, such as 1e-3 or smaller
	double MiniChange; // Control the prcision of shrink phase (replicator equation), usually set to 1e-3 or smaller
	double Alpha;      // The smallest probability of a vertex, smaller than it will be consider of probability zero, usually set to 0.1/NumOfVerticesInMaxima
	double MinValue;   // Smaller than it will be consider to be zero, usually set to a very small value, such as 1e-10
}Parameter;

class SEA
{
public:	
	IIDVector AMElems;
	IntArray  ExpNeighbors;
	ProPointVector StartPoints;
	Parameter Para;

	IDArray Neighbors;
	IntVector VertexMarkVector,CurSubgraph;	

	DoubleArray Maxima;
	IntArray Support;
	DoubleVector FValue;
	bool ReadData(const mxArray* prhs[],int nrhs);
	void BuildNeighbors(int Mark);
	void Run(void);
	void SEAProcedure(int StartIndex);
	void InitDataStructure(void);

	bool ExpansionPhase(IntVector& CurSupport,DoubleVector& CurPoint,double& CurValue);
	void ShrinkPhase(IntVector& CurSupport,DoubleVector& CurPoint,double& CurValue);
	void QuickSortL(IDVector& data,int left,int right);
	void OutputResult(int& nlhs,mxArray* plhs[]);		
	double ComputeFunValue(IntVector& CurSupport, DoubleVector& CurPoint);	
};


bool SEA::ReadData(const mxArray *prhs[],int nrhs)
{
	if(nrhs!=4)
		return false;

	//read sparse adjacency matrix elements
	int NumOfEntries = mxGetM(prhs[0]);
	double* pData = mxGetPr(prhs[0]);
	for(int i=0;i<NumOfEntries;i++)
	{
		IntIntDouble Iid;
		Iid.I[0] = (int) pData[i]-1;
		Iid.I[1] = (int) pData[NumOfEntries+i]-1;
		Iid.D    = pData[2*NumOfEntries+i];

		if(Iid.I[0]<Iid.I[1])
			AMElems.push_back(Iid);
	}

	int NumOfVertices = mxGetM(prhs[1]);
	for(int i=0;i<NumOfVertices;i++)
	{
		IntVector Nei;
		mxArray* pCell = mxGetCell(prhs[1],i);
		int NumOfNei = mxGetM(pCell);
		pData = mxGetPr(pCell);
		for(int j=0;j<NumOfNei;j++)
			Nei.push_back((int)pData[j]-1);
		ExpNeighbors.push_back(Nei);
	}

	int NumOfStartPoints = mxGetM(prhs[2]);
	for(int i=0;i<NumOfStartPoints;i++)
	{
		ProPoint Pp;
		mxArray* pCell = mxGetCell(prhs[2],i);
		pData = mxGetPr(pCell);		
		int NumOfElems = mxGetM(pCell);
		for(int j=0;j<NumOfElems;j++)
		{
			Pp.Support.push_back((int)pData[j]-1);			
			Pp.Prob.push_back(pData[NumOfElems+j]);
		}
		StartPoints.push_back(Pp);
	}	

	pData = mxGetPr(prhs[3]);
	int NumOfParas = mxGetM(prhs[3]);
	if(NumOfParas<1)
		return false;

	Para.NumOfVertices   = (int) pData[0];
	if(NumOfParas<6)
		Para.MinValue    = 0.0000000001;
	else
		Para.MinValue    = pData[5];
    if(NumOfParas<5)
		Para.MinIncrease = 0.001;
	else
		Para.MinIncrease = pData[4];
	if(NumOfParas<4)
		Para.MiniChange  = 0.00001;
	else
		Para.MiniChange  = pData[3];
	if(NumOfParas<3)
		Para.Alpha       = 0.001;
	else
		Para.Alpha       = pData[2];	
	if(NumOfParas<2)
		Para.K           = Para.NumOfVertices;
	else
		Para.K           = (int)pData[1];
	
	return true;
}


//Mark:1 Need to build expansion neighbors
void SEA::BuildNeighbors(int Mark)
{
	IDVector Idv;
	for(int i=0;i<Para.NumOfVertices;i++)
		Neighbors.push_back(Idv);

	if(Mark==1)
	{
		IntVector Iv;
		for(int i=0;i<Para.NumOfVertices;i++)
			ExpNeighbors.push_back(Iv);
	}

	if(Mark==0)
	{
		for(int i=0;i<AMElems.size();i++)
		{
			IntIntDouble Iid = AMElems[i];
			IntDouble Id;
			Id.D = Iid.D;
			Id.I = Iid.I[1];
			Neighbors[Iid.I[0]].push_back(Id);
			Id.I = Iid.I[0];
			Neighbors[Iid.I[1]].push_back(Id);
		}		
	}
	else
	{
		for(int i=0;i<AMElems.size();i++)
		{
			IntIntDouble Iid = AMElems[i];
			IntDouble Id;
			Id.D = Iid.D;
			Id.I = Iid.I[1];
			Neighbors[Iid.I[0]].push_back(Id);	
			Id.I = Iid.I[0];
			Neighbors[Iid.I[1]].push_back(Id);

            ExpNeighbors[Iid.I[0]].push_back(Iid.I[1]);
			ExpNeighbors[Iid.I[1]].push_back(Iid.I[0]);
		}			
	}	
}

void SEA::InitDataStructure(void)
{
	if(ExpNeighbors.size()==0)
		BuildNeighbors(1);
	else
		BuildNeighbors(0);	

	for(int i=0;i<Para.NumOfVertices;i++)
	{
		VertexMarkVector.push_back(-1);
		CurSubgraph.push_back(-1);
	}
}

void SEA::Run(void)
{
	InitDataStructure();

	//evolve from every starting vertex
	int NumOfStartPoints   = StartPoints.size();
	for(int i=0;i<NumOfStartPoints;i++)
	{
		SEAProcedure(i);
		mexPrintf("Evolve from the %d start points\n",i);
	}
}

void SEA::QuickSortL(IDVector& data,int left,int right)
{
	int l_hold, r_hold, mid;
	IntDouble pivot;		
    l_hold = left;
    r_hold = right;
    mid=(left+right)/2;	
	pivot = data[left];
	
    while (left < right)
    {
	  while ((data[right].D<pivot.D) && (left < right))
			right--;
      if (left != right)
      {
		  data[left] = data[right];		
          left++;
      }
	  while ((data[left].D>= pivot.D) && (left < right))
          left++;
      if (left != right)
      {
		  data[right] = data[left];		 
          right--;
      }
    }

	data[left] = pivot;	
    
	mid=left;
    left = l_hold;
    right = r_hold;
    if (left < mid)
		QuickSortL(data, left, mid-1);
    if (right > mid)
		QuickSortL(data, mid+1, right);
}

bool SEA::ExpansionPhase(IntVector& CurSupport,DoubleVector& CurPoint,double& CurValue)
{	
	//find all neighbors,excluding vertices in current subgraph		
	for(int i=0;i<Para.NumOfVertices;i++)
		VertexMarkVector[i] = -1;
	
    IntVector ReverseMap;
	for(int i=0;i<CurSupport.size();i++)
	{
		int CurIndex = CurSupport[i];
		for(int j=0;j<ExpNeighbors[CurIndex].size();j++)
		{
			int NeiIndex = ExpNeighbors[CurIndex][j];	
			if(CurSubgraph[NeiIndex]>=0)
				continue;
			if(VertexMarkVector[NeiIndex]>=0)
				continue;			
			VertexMarkVector[NeiIndex] = ReverseMap.size();
			ReverseMap.push_back(NeiIndex);					
		}		
	}
	
	int NumOfNeighbors = ReverseMap.size();
	if(NumOfNeighbors==0)
		return true;

	//calcualte the reward at every neighbor vertex
	IIDVector SubAMElems;
	for(int i=0;i<CurSupport.size();i++)
	{
		int CurIndex = CurSupport[i];
		for(int j=0;j<Neighbors[CurIndex].size();j++)
		{
			IntDouble Id = Neighbors[CurIndex][j];
			int LIndex = VertexMarkVector[Id.I];
			if(LIndex<0)
				continue;			
			IntIntDouble Iid;
			Iid.I[0] = i;
			Iid.I[1] = LIndex;
			Iid.D    = Id.D;
			SubAMElems.push_back(Iid);
		}
	}
	DoubleVector Rewards;
	for(int i=0;i<NumOfNeighbors;i++)
		Rewards.push_back(0);
	for(int i=0;i<SubAMElems.size();i++)
	{
		IntIntDouble Iid = SubAMElems[i];
		Rewards[Iid.I[1]] += Iid.D*CurPoint[Iid.I[0]];
	}

	//find candidate vertices
	IDVector CandidateVertices;
	for(int i=0;i<NumOfNeighbors;i++)
	{
		Rewards[i] -= CurValue;
		if(Rewards[i]>Para.MinValue)
		{
			IntDouble Id;
			Id.I = ReverseMap[i];
			Id.D = Rewards[i];
			CandidateVertices.push_back(Id);
		}
	}

	//limit the number of added vertices
	int NumOfCandidates = CandidateVertices.size();	
	if(NumOfCandidates>Para.K)
	{		
		QuickSortL(CandidateVertices,0,CandidateVertices.size()-1);
		NumOfCandidates = Para.K;
	}	

	for(int i=0;i<Para.NumOfVertices;i++)
		VertexMarkVector[i] = -1;	
	for(int i=0;i<NumOfCandidates;i++)		
			VertexMarkVector[CandidateVertices[i].I] = i;	

	double Vars = 0;
	double Varsigma = 0;
	for(int i=0;i<NumOfCandidates;i++)
	{
		double d = CandidateVertices[i].D;
		Vars += d;
		Varsigma += d*d;
	}

	//it is still a KKT point, thus stop
	if(Vars<Para.Alpha)
		return true;

	//compute omega
    double Omega = 0;
	for(int i=0;i<NumOfCandidates;i++)
	{
		int CurIndex = CandidateVertices[i].I;
		for(int j=0;j<Neighbors[CurIndex].size();j++)
		{
			int NeiIndex = Neighbors[CurIndex][j].I;
			int LIndex = VertexMarkVector[NeiIndex];
			if(i>LIndex)
				continue;
			if(LIndex<0)
				continue;				
			Omega += Neighbors[CurIndex][j].D*CandidateVertices[i].D*CandidateVertices[LIndex].D;
		}
	}	
	Omega *= 2;

	IntVector NewSupport;
	DoubleVector b; //the update direction
	for(int i=0;i<CurSupport.size();i++)
	{
		NewSupport.push_back(CurSupport[i]);
		b.push_back(-CurPoint[i]*Vars);
	}
    for(int i=0;i<NumOfCandidates;i++)
	{
		NewSupport.push_back(CandidateVertices[i].I);
		CurPoint.push_back(0);
		b.push_back(CandidateVertices[i].D);		
	}

	//update by expansion vector
	double Coeff2 = CurValue*Vars*Vars+2*Vars*Varsigma-Omega;
	double Bestt = 1.0/Vars;
	if(Coeff2>0)
	{
		double t= Varsigma/Coeff2;
		if(t<Bestt)
			Bestt = t;
	}	
	for(int i=0;i<CurPoint.size();i++)
		CurPoint[i] += Bestt*b[i];
	
	CurValue += -Coeff2*Bestt*Bestt+2*Varsigma*Bestt;
	CurSupport = NewSupport;			
	
	for(int i=0;i<Para.NumOfVertices;i++)
		CurSubgraph[i] = -1;
	for(int i=0;i<CurSupport.size();i++)
		CurSubgraph[CurSupport[i]] = i;	
	return false;	
}

void SEA::ShrinkPhase(IntVector& CurSupport,DoubleVector& CurPoint,double& CurValue)
{
	//first construct subgraph
	int NumOfVertices = CurSupport.size();
	if(NumOfVertices==1)
		return;

	IIDVector SubAMElems;
	for(int i=0;i<CurSupport.size();i++)
	{
		int CurIndex = CurSupport[i];
		for(int j=0;j<Neighbors[CurIndex].size();j++)
		{
			IntDouble Id = Neighbors[CurIndex][j];			
			int LIndex = CurSubgraph[Id.I];
			if(LIndex<0)
				continue;

			IntIntDouble Iid;
			Iid.I[0] = i;
			Iid.I[1] = LIndex;
			Iid.D    = Id.D;
			SubAMElems.push_back(Iid);
		}
	}

	//second, evolving to KKT points by replicator equation
	DoubleVector Rewards;
	for(int i=0;i<NumOfVertices;i++)
		Rewards.push_back(0);

	double PreValue = 0;
	while(true)
	{		
		for(int i=0;i<NumOfVertices;i++)
			Rewards[i] = 0;
		for(int i=0;i<SubAMElems.size();i++)
		{
			IntIntDouble Iid = SubAMElems[i];
			Rewards[Iid.I[0]] += Iid.D*CurPoint[Iid.I[1]];			
		}

		CurValue = 0;
		for(int i=0;i<NumOfVertices;i++)
		{
			CurPoint[i] *= Rewards[i];
			CurValue += CurPoint[i];
		}
		for(int i=0;i<NumOfVertices;i++)
			CurPoint[i] /= CurValue;

		double Ratio = (CurValue - PreValue)/PreValue;
		if(Ratio<Para.MiniChange)
			break;		
		PreValue = CurValue;
	}

	bool IsChange = false;
	IntVector NewSupport;
	DoubleVector NewPoint;
	double Sum = 0;
	for(int i=0;i<CurPoint.size();i++)
	{
		if(CurPoint[i]>Para.Alpha)
		{
			NewSupport.push_back(CurSupport[i]);
			NewPoint.push_back(CurPoint[i]);
			Sum += CurPoint[i];			
		}	
		else		
			IsChange = true;		
	}

	if(IsChange)
	{
		for(int i=0;i<NewPoint.size();i++)
			NewPoint[i] /= Sum;

		CurPoint   = NewPoint;
		CurSupport = NewSupport;
	}		
}

void SEA::SEAProcedure(int StartIndex)
{	
	DoubleVector CurPoint;
	IntVector    CurSupport;
	double       CurValue;

	//initialization
	CurSupport = StartPoints[StartIndex].Support;
	CurPoint   = StartPoints[StartIndex].Prob;
	CurValue   = ComputeFunValue(CurSupport,CurPoint);
	for(int i=0;i<Para.NumOfVertices;i++)
		CurSubgraph[i] = -1;
	for(int i=0;i<CurSupport.size();i++)
		CurSubgraph[CurSupport[i]] = i;

	//SEA iteration	
	while(true)
	{		
		double PreValue = CurValue;	

		//Expansion Phase				
		bool IsMaxima = ExpansionPhase(CurSupport,CurPoint,CurValue);
		if(IsMaxima)
			break;		
		
		//Shrink Phase
		ShrinkPhase(CurSupport,CurPoint,CurValue);		       

		//to avoid possible "expand" and "shrink" oscillations
		if((CurValue-PreValue)<Para.MinIncrease)
			break;		
	}

	Maxima.push_back(CurPoint);
	Support.push_back(CurSupport);
	FValue.push_back(CurValue);	
}

void SEA::OutputResult(int& nlhs,mxArray* plhs[])
{
	nlhs = 3;

	int Num = Maxima.size();
	plhs[0] = mxCreateCellMatrix(Num,1);
	for(int i=0;i<Num;i++)
	{
		int Num1 = Maxima[i].size();
		mxArray* pCell = mxCreateDoubleMatrix(Num1,1,mxREAL);
		double* pData  = mxGetPr(pCell);
		for(int j=0;j<Num1;j++)
			pData[j] = Maxima[i][j];
		mxSetCell(plhs[0],i,pCell);
	}

	plhs[1] = mxCreateCellMatrix(Num,1);
	for(int i=0;i<Num;i++)
	{
		int Num1 = Support[i].size();
		mxArray* pCell = mxCreateDoubleMatrix(Num1,1,mxREAL);
		double* pData  = mxGetPr(pCell);
		for(int j=0;j<Num1;j++)
			pData[j] = Support[i][j]+1;
		mxSetCell(plhs[1],i,pCell);
	}

	plhs[2] = mxCreateDoubleMatrix(Num,1,mxREAL);
	double* pData = mxGetPr(plhs[2]);
	for(int i=0;i<Num;i++)
		pData[i] = FValue[i];
}


double SEA::ComputeFunValue(IntVector& CurSupport, DoubleVector& CurPoint)
{
	for(int i=0;i<Para.NumOfVertices;i++)
		VertexMarkVector[i] = -1;
	for(int i=0;i<CurSupport.size();i++)
		VertexMarkVector[CurSupport[i]] = i;

	IIDVector SubAMElems;
	for(int i=0;i<CurSupport.size();i++)
	{
		int CurIndex = CurSupport[i];
		for(int j=0;j<Neighbors[CurIndex].size();j++)
		{
			int NeiIndex = Neighbors[CurIndex][j].I;
			int LIndex = VertexMarkVector[NeiIndex];
			if(i>LIndex)
				continue;
			if(LIndex<0)
				continue;			

			IntIntDouble Iid;
			Iid.I[0] = i;
			Iid.I[1] = LIndex;
			Iid.D    = Neighbors[CurIndex][j].D;
			SubAMElems.push_back(Iid);
		}
	}

	double CurValue = 0;
	for(int i=0;i<SubAMElems.size();i++)
	{
		IntIntDouble Iid = SubAMElems[i];
		CurValue += Iid.D*CurPoint[Iid.I[0]]*CurPoint[Iid.I[1]];
	}

	return 2*CurValue;	
}


void mexFunction(int nlhs, mxArray *plhs[],int nrhs, const mxArray *prhs[])
{
	SEA Sea;
	if(!Sea.ReadData(prhs,nrhs))
		return;
	Sea.Run();
	Sea.OutputResult(nlhs,plhs);
}

