#include <mex.h>
#include <math.h>
#include <vector>
#include <map>
#include <set>
#include <windows.h>
using namespace std;

/*
  [Maxima,FValue] = ReplicatorEquation(AdjMatrixElems,Para);
  
*/

typedef vector<double> DoubleVector;
typedef vector<DoubleVector> DoubleArray;

typedef vector<int> IntVector;
typedef vector<IntVector> IntArray;

typedef struct ProPoint_TYP{
	IntVector Support;
	DoubleVector Prob;
}ProPoint;
typedef vector<ProPoint> ProPointVector;

typedef struct IntDouble_TYP{
	int I;
	double D;
}IntDouble;
typedef vector<IntDouble> IDVector;
typedef vector<IDVector>  IDArray;

typedef struct IntIntDouble_TYP{
	int I[2];
	double D;
}IntIntDouble;
typedef vector<IntIntDouble> IIDVector;

//seem many parameters, but in fact only the set need to be set, others can be constant.
typedef struct Parameter_TYP{	
	int NumOfVertices; 		
	double MiniChange; 		
}Parameter;

class ReplicatorEquation
{
public:
     IIDVector AMElems;
	 Parameter Para;

	 DoubleVector Maxima;
	 double FValue;
	 bool ReadData(int nrhs,const mxArray* prhs[]);	
	 double ComputeFunValue(void);
	 void Run(void);
	 void OutputResult(int& nlhs,mxArray* plhs[]);
};

bool ReplicatorEquation::ReadData(int nrhs,const mxArray* prhs[])
{
	if(nrhs!=2)
		return false;

	//read sparse adjacency matrix elements
	int NumOfEntries = mxGetM(prhs[0]);
	double* pData = mxGetPr(prhs[0]);
	for(int i=0;i<NumOfEntries;i++)
	{
		IntIntDouble Iid;
		Iid.I[0] = (int) pData[i]-1;
		Iid.I[1] = (int) pData[NumOfEntries+i]-1;
		Iid.D    = pData[2*NumOfEntries+i];

		if(Iid.I[0]<Iid.I[1])
			AMElems.push_back(Iid);
	}

	pData = mxGetPr(prhs[1]);
	Para.NumOfVertices = (int)pData[0];	
	Para.MiniChange    = pData[1];	
	return true;
}

double ReplicatorEquation::ComputeFunValue(void)
{
	double f = 0;
	int NumOfEntry = AMElems.size();
	for(int i=0;i<NumOfEntry;i++)
	{
		IntIntDouble Iid = AMElems[i];
		f += Iid.D*Maxima[Iid.I[0]]*Maxima[Iid.I[1]];
	}
	f *= 2;
	return f;
}

void ReplicatorEquation::Run(void)
{
	int NumOfVertices = Para.NumOfVertices;
	DoubleVector Deri;
	for(int i=0;i<NumOfVertices;i++)
	{
		Maxima.push_back(1.0/NumOfVertices);
		Deri.push_back(0);
	}

	FValue = ComputeFunValue();
	double PreValue = 0;
	while(true)
	{			
		for(int i=0;i<NumOfVertices;i++)
			Deri[i] = 0;

		for(int i=0;i<AMElems.size();i++)
		{
			IntIntDouble Iid = AMElems[i];
			Deri[Iid.I[0]] += Maxima[Iid.I[1]]*Iid.D;
			Deri[Iid.I[1]] += Maxima[Iid.I[0]]*Iid.D;
		}

		for(int i=0;i<NumOfVertices;i++)
			Maxima[i] *= Deri[i];
		FValue = 0;
		for(int i=0;i<NumOfVertices;i++)
			FValue += Maxima[i];

		for(int i=0;i<NumOfVertices;i++)
			Maxima[i] /= FValue;

		double Ratio = (FValue-PreValue)/PreValue;
		mexPrintf("PreValue:%lf,CurValue:%lf,Ratio:%lf\n",PreValue,FValue,Ratio);
		if(Ratio<Para.MiniChange)
			break;
		PreValue = FValue;
	}	
}

void ReplicatorEquation::OutputResult(int& nlhs,mxArray* plhs[])
{
	nlhs = 2;
	plhs[0] = mxCreateDoubleMatrix(Para.NumOfVertices,1,mxREAL);
	double* pData = mxGetPr(plhs[0]);
	for(int i=0;i<Para.NumOfVertices;i++)
		pData[i] = Maxima[i];

	plhs[1] = mxCreateDoubleMatrix(1,1,mxREAL);
	pData = mxGetPr(plhs[1]);
	pData[0] = FValue;
}

void mexFunction(int nlhs, mxArray *plhs[],int nrhs, const mxArray *prhs[])
{
	ReplicatorEquation Re;
	if(!Re.ReadData(nrhs,prhs))
		return;
	Re.Run();
	Re.OutputResult(nlhs,plhs);
}