/*---------------------------------------------------------------------------
File:         			Environment.java
Package:                        JFLAP Version 2.0
Author:				Magda & Octavian Procopiuc V1.0 07/15/96
                                Eric Gramond V2.0 07/22/97 
 
Description of Contents:	Contains class Environment.
				
--------------------------------------------------------------------------*/

/* 
 * Susan H. Rodger, Magda Procopiuc, Octavian Procopiuc, Eric Gramond
 * Computer Science Department
 * Duke University
 * June 1997
 * Supported by National Science Foundation DUE-9596002 and DUE-9555084.
 *
 * Copyright (c) 1997
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the author.  The name of the author may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package flap;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.io.*;
import java.net.URL;
import java.util.*;

/**
 * This is the main window associated with each machine that the user
 * opens. It contains the Desktop in the middle -- where the machine 
 * is displayed and 
 * edited --, a status bar at the bottom, an input string bar  
 * above the Desktop, and a menu bar at the top. 
 * It also contains the actual machine.
 *
 * @see		flap.Desktop
 * @author	Magda & Octavian Procopiuc
 * @version	1.0, 07/15/96
 */
public class Environment extends Frame implements ActionListener, WindowListener, MouseListener, KeyListener, ItemListener{

  /**
   * The type of the machine.
   */
  public int		type;
  /**
   * The name of the file containing the current machine.
   */
  public String		fileName = NO_FILE_NAME;
  /**
   * The desktop.
   */
  public Desktop	d;
 /**
  * Other desktop used by the TO_DFA FSA's to store the DFA to compare with
  */
  public FSA        dfa; 
 /**
   * The machine.
   */
  public Machine	a;
  /**
   * The input string field.
   */
  public TextField	inputField;
  /** 
    * the label displayed to the left of the textfield
    */
  public Label           inputName;
  /**
   * The input string field for the second tape for TM2.
   */
  public TextField	inputField2;
  /**
    * the isDone button for minimize tree
    */
  public Button isMinimizeDone;
  /**
    * the show button for minimize tree
    */
  public Button showAllMinimize;
  /** 
   * The NFA to DFA button for FSA
   */
  public MenuItem       minimize;
  public MenuItem       nfaToDfa;
  /** 
    * The isDone button for a to-dfa window
    */
  public MenuItem       minIsDone;
  public MenuItem       isDone;
  /**
    * The show button for a to-dfa window
    */
  public MenuItem       minShow;
  public MenuItem       show;
  /** 
    * The expand state button for a to-dfa window
    */
  public  MenuItem       minExpand;
  public  MenuItem       expand;

  private MenuItem       newItem;
  private MenuItem       openItem;
  private MenuItem       saveItem;
  private MenuItem       saveAsItem;
  private MenuItem       tabularItem;

  /**
    * The checkbox to hide or display name labels for to-dfa window
    */
  public CheckboxMenuItem showNameLabels;
  /**
    * The Environment that generated this Environment
    */
  public Environment envSource;
  /**
    * the Environment that this Environment generated. When this is set the window is frozen
    */
  public Environment envTarget;
  /**
    * the environment with the tree for minimize machines
    */
  public Environment oldTree;
  /**
   * The bottom status line.
   */
  public Label		statusLine;
  /**
   * The acceptance choice for PDA.
   */
  public Choice		acceptanceChoice;
  /**
   * The Container which opened this Environment.
   * Can be either MainMenu or FLAPApplet.
   */
  protected MainMenu	mainmenu;
  /**
   * The current directory for opening/saving.
   */
  public String		currentDir = null;
  /**
   * When isApplet is set, some items in the menu bar are disabled.
   */
  public boolean	isApplet = false;
  boolean 	done = false;

  


  public final static String    SHOW_GRAM               = "Show Grammar";
  public final static String	CLOSE_STR		= "Close";
  public final static String	QUIT_STR		= "Quit flap";
  public final static String    DONE_STR                = "Check if done";
  public final static String    TO_DFA_STR              = "Convert to DFA";
  public final static String    SHOW_STR                = "Show";
  public final static String    EXPAND_STATE_STR        = "Expand a state";
  public final static String    SHOW_LABEL_STR          = "Show Name Labels";
  public final static String    MINIMIZE_STR            = "Minimize";
  public final static String    SHOW_UNREACHABLE_STR    = "Show unreachable states";
  public final static String	HLP_STATE_STR		= "States";
  public final static String	HLP_TRANSITION_STR	= "Transitions";
  public final static String	HLP_OVERVIEW_STR	= "Overview";
  public final static String	HLP_UPPERWINDOW_STR	= "Step Run Window";
  public final static String	HLP_TRACEWINDOW_STR	= "Trace Window";
  public final static String	HLP_MENUBAR_STR		= "Menu Bar";
  public final static String    HLP_NFATODFA_STR        = "NFA to DFA";
  public final static String    HLP_MINIMIZE_STR        = "Minimize a DFA";
  public final static String    AUTOMATON_IS_DETER_STR  = "The automaton is deterministic.";
  public final static String    AUTOMATON_IS_NON_DETER_STR = "The automaton is non deterministic.";
  public final static String    DFA_IS_CORRECT_STR      = " The DFA is correct.";
  public final static String    DONE_MINIMIZE_STR       = "Done";
  public final static String    SHOW_ALL_MINIMIZE_STR   = "Show All";

  public final static String	NO_FILE_NAME		= "(no name)";

  final static String	SAVE_STRING		= "Save";
  final static String	NOTSAVE_STRING		= "Don't Save";
  final static String	CANCEL_STRING		= "Cancel";
  final static String[]	SAVE_BUTTONS		= {SAVE_STRING, NOTSAVE_STRING, CANCEL_STRING};

  final static int	NEW         = 0;
  final static int	OPEN        = 1;
  final static int	FASTRUN     = 2;
  final static int	PRINT       = 3;
  final static int	CLOSE       = 4;
  final static int	SHOWNONDETS = 5;
  final static int      TO_DFA      = 6;
  final static int      MINIMIZE    = 7;
  
  private Menu		filemenu, optionsmenu, runmenu, helpmenu;

  public Environment(int type, String aFileName, MainMenu mainmenu, boolean isApplet) {
    super();
    setLocation(mainmenu.getLocation().x+40, 20);
    this.isApplet = isApplet;
    this.mainmenu = mainmenu;
    this.type = type;
    this.addWindowListener(this);
    this.addKeyListener(this);
    setBackground(Params._e_backcolor);
    if(type == Machine.MTC)
      d = new MinimizeDesktop(type);
    else  d = new Desktop(type);
    add("Center", d);
    switch (type) {
    case Machine.FSA:
      a = new FSA(d);
      break;
    case Machine.PDA:
      a = new PDA(d);
      break;
    case Machine.TM1:
      a = new TM1(d);
      break;
    case Machine.TM2:
      a = new TM2(d);
      break;
    case Machine.MTC:
      a = new TreeMachine(d);
      break;
    default:
      setVisible(false);
      System.err.println("Internal error. Please report.");
      return;
    }
    statusLine = new Label("", Label.LEFT);
    add("South", statusLine);
    isMinimizeDone = new Button(DONE_MINIMIZE_STR);
    isMinimizeDone.addActionListener(this);
    showAllMinimize = new Button(SHOW_ALL_MINIMIZE_STR);
    showAllMinimize.addActionListener(this);
    Panel inputLine = new Panel();
    inputLine.add(inputName = new Label((type == Machine.TM2) ? "Input (tape 1):" : "Input String: "));
    inputLine.add(inputField = new TextField("", 
		  type == Machine.TM2 ? 23 : Params._e_inputfieldlength));
   		  
    if (type == Machine.FSA) {
      inputLine.add(new Label("      "));
    } else if (type == Machine.PDA) {
      inputLine.add(new Label("      Acceptance: "));
      inputLine.add(acceptanceChoice = new Choice());
      acceptanceChoice.addItem(PDA.EMPTY_STACK);
      acceptanceChoice.addItem(PDA.FINAL_STATE);
      acceptanceChoice.addItem(PDA.FINAL_AND_EMPTY);
      acceptanceChoice.select(((PDA) a).acceptanceType);
      acceptanceChoice.addItemListener(this);
    } else if (type == Machine.TM2) {
      inputLine.add(new Label("Input (tape 2):"));
      inputLine.add(inputField2 = new TextField("", 23));
    }
    inputLine.add(showAllMinimize);
    showAllMinimize.setVisible(false); 
    inputLine.add(isMinimizeDone);
    isMinimizeDone.setVisible(false);

    add("North", inputLine);
    putMenuBar();
    updateStatus("Mail any bugs or comments to: rodger@cs.duke.edu");
    if (!aFileName.equals(NO_FILE_NAME)) {
      String str = (aFileName.endsWith("."+strType())) ? aFileName: aFileName + "."+ strType();
      fileName = str;
      try {
        open(aFileName);
      }
      catch (Exception e) {
        updateStatus("Could not open file " + str + ". New file.");
      }
    }
    setIconImage(Params._img_icon);
  }

  public Dimension getPreferredSize() {
    return new Dimension(Params._e_width, Params._e_height);
  }

  /**
   * Constructs the menu bar.
   */
  private void putMenuBar() {
    MenuBar 	mb = new MenuBar();
    Menu	m;
    MenuItem	item;

    mb.setFont(Params._m_font);
    m = new Menu("File");
    filemenu = m;
    m.setFont(Params._m_font);
    m.add(newItem = new MenuItem("New", new MenuShortcut('n')));
    newItem.addActionListener(this);
    m.add(openItem = new MenuItem("Open...", new MenuShortcut('o')));
    if (isApplet) openItem.setEnabled(false);
    openItem.addActionListener(this);
    m.add(saveItem = new MenuItem("Save", new MenuShortcut('s')));
    if (isApplet) saveItem.setEnabled(false);
    saveItem.addActionListener(this);
    m.add(saveAsItem = new MenuItem("Save as...", new MenuShortcut('S',true)));
    if (isApplet) saveAsItem.setEnabled(false);
    saveAsItem.addActionListener(this);
    m.addSeparator();
    Menu printMenu = new Menu("Print...");
    printMenu.add(tabularItem = new MenuItem("tabular", new MenuShortcut('P', true)));
    if (isApplet) tabularItem.setEnabled(false);
    tabularItem.addActionListener(this);
    printMenu.add(item = new MenuItem("postscript", new MenuShortcut('p')));
    if (isApplet) item.setEnabled(false);
    item.addActionListener(this);
    m.add(printMenu);
    m.addSeparator();
    m.add(item = new MenuItem(CLOSE_STR, new MenuShortcut('c')));
    item.addActionListener(this);
    m.add(item = new MenuItem(QUIT_STR, new MenuShortcut('q')));
    if (isApplet) item.setEnabled(false);
    item.addActionListener(mainmenu);
    mb.add(m);

    m = new Menu("Run");
    runmenu = m;
    m.setFont(Params._m_font);
    m.add(item = new MenuItem("Step run", new MenuShortcut('r')));
    item.addActionListener(this);
    m.add(item = new MenuItem("Fast run", new MenuShortcut('R', true)));
    item.addActionListener(this);
    mb.add(m);

    m = new Menu("Options");
    optionsmenu = m;
    m.setFont(Params._m_font);
    m.add(item = new MenuItem("Re-label states", new MenuShortcut('l')));
    item.addActionListener(this);
    m.add(item = new MenuItem("Show nondeterministic states", new MenuShortcut('w')));
    item.addActionListener(this);
    Menu dfaMenu = new Menu("NFA to DFA");
    if(type == Machine.FSA) {
      m.add(item = new MenuItem(SHOW_GRAM, new MenuShortcut('g')));
      item.addActionListener(this);
      m.add(dfaMenu);
      dfaMenu.add(nfaToDfa = new MenuItem(TO_DFA_STR, new MenuShortcut('f')));
      nfaToDfa.addActionListener(this);
      dfaMenu.add(isDone = new MenuItem(DONE_STR, new MenuShortcut('d')));
      isDone.addActionListener(this);
      isDone.setEnabled(false);
      dfaMenu.add(show = new MenuItem(SHOW_STR, new MenuShortcut('h')));
      show.addActionListener(this);
      show.setEnabled(false);
      dfaMenu.add(expand = new MenuItem(EXPAND_STATE_STR, new MenuShortcut('e')));
      expand.addActionListener(this);
      expand.setEnabled(false);
    }
 
    m.add(item = new MenuItem(SHOW_UNREACHABLE_STR, new MenuShortcut('u')));
    item.addActionListener(this);    
    if(type != Machine.FSA) item.setEnabled(false);

    Menu minMenu = new Menu("Minimize");
    if(type == Machine.FSA) {
      m.add(minMenu);
      minMenu.add(minimize = new MenuItem(MINIMIZE_STR, new MenuShortcut('m')));
      minimize.addActionListener(this);
      minMenu.add(minIsDone = new MenuItem(DONE_STR, new MenuShortcut('d')));
      minIsDone.addActionListener(this);
      minIsDone.setEnabled(false);
      minMenu.add(minShow = new MenuItem(SHOW_STR, new MenuShortcut('h')));
      minShow.addActionListener(this);
      minShow.setEnabled(false);
      minMenu.add(minExpand = new MenuItem(EXPAND_STATE_STR, new MenuShortcut('e')));
      minExpand.addActionListener(this);
      minExpand.setEnabled(false);
    }

    if (isApplet) {
      m.add(item = new MenuItem("Load example"));
      item.addActionListener(this);
    }

    m.addSeparator();
    if(type == Machine.FSA) {
      m.add(showNameLabels = new CheckboxMenuItem(SHOW_LABEL_STR, false));
      showNameLabels.addItemListener(this);
      showNameLabels.setEnabled(false);
    }
    m.add(item = new CheckboxMenuItem("Auto re-label states"));
    ((CheckboxMenuItem) item).setState(d.autoRelabel);
    ((CheckboxMenuItem) item).addItemListener(this);

    mb.add(m);

    m = new Menu("Help");
    helpmenu = m;
    m.setFont(Params._m_font);
    m.add(item = new MenuItem("About..."));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_OVERVIEW_STR));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_STATE_STR));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_TRANSITION_STR));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_UPPERWINDOW_STR));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_TRACEWINDOW_STR));
    item.addActionListener(this);
    m.add(item = new MenuItem(HLP_MENUBAR_STR));
    item.addActionListener(this);
    if(type == Machine.FSA) {
      m.add(item =new MenuItem(HLP_NFATODFA_STR));
      item.addActionListener(this);
      m.add(item = new MenuItem(HLP_MINIMIZE_STR));
      item.addActionListener(this);
    }

    mb.setHelpMenu(m);	// setHelpMenu is not working on all versions...


    setMenuBar(mb);
  }	// end of method putMenuBar.

  public void windowClosing(WindowEvent e) {
    Thread actionThread;
    actionThread = new DialogThread(this, CLOSE);
    actionThread.start();
  }
  public void windowOpened(WindowEvent e) 
  {}
  public void windowIconified(WindowEvent e)  
  {}
  public void windowDeiconified(WindowEvent e) 
  {}
  public void windowClosed(WindowEvent e)  
  {}
  public void windowActivated(WindowEvent e)
  {}
  public void windowDeactivated(WindowEvent e)
  {}

  public void mousePressed(MouseEvent e) 
  {}
  public void mouseReleased(MouseEvent e) 
  {
    updateStatus();
  }
  public void mouseEntered(MouseEvent e) 
  {}
  public void mouseExited(MouseEvent e) 
  {}
  public void mouseClicked(MouseEvent e)
  {}

  public void keyPressed(KeyEvent e) 
  {}
  public void keyReleased(KeyEvent e) 
  {
    updateStatus();
  }
  public void keyTyped(KeyEvent e)
  {}

  public void itemStateChanged(ItemEvent e){
    Thread actionThread;
    if (e.getSource() instanceof Choice){
	((PDA) a).acceptanceType = acceptanceChoice.getSelectedItem();
    } 
    else if (e.getSource() instanceof CheckboxMenuItem){
      String arg = ((CheckboxMenuItem)(e.getSource())).getLabel();
      if (arg.equals("Auto re-label states")){
	d.autoRelabel = ((CheckboxMenuItem) e.getSource()).getState();
	d.relabel();
      }
      else if (e.getSource() == showNameLabels) {
	d.showDfaLabels = ((CheckboxMenuItem) e.getSource()).getState();
	if(dfa != null) 
	  dfa.d.showDfaLabels = d.showDfaLabels;
	d.repaint();
      }
    }
  }

  public void actionPerformed(ActionEvent e){
    Thread actionThread;
    if (e.getSource() instanceof MenuItem) {
	String arg = ((MenuItem)(e.getSource())).getLabel();
	if (arg.equals("New")){
	    actionThread = new DialogThread(this, NEW);
	    actionThread.start(); } 
	else if ("Open...".equals(arg)) {
	    actionThread = new DialogThread(this, OPEN);
	    actionThread.start(); }     
	else if ("Save as...".equals(arg)) 
	  saveAs();
	else if ("Save".equals(arg))
	  save();
	else if (CLOSE_STR.equals(arg)) {
	  actionThread = new DialogThread(this, CLOSE);
	  actionThread.start(); }     
	else if ("Step run".equals(arg))
	  stepRun();
	else if ("Fast run".equals(arg)) {
	  actionThread = new DialogThread(this, FASTRUN);
	  actionThread.start(); }     
	else if ("Re-label states".equals(arg))
	  relabel();
	else if ("Show nondeterministic states".equals(arg)) {
	  actionThread = new DialogThread(this, SHOWNONDETS);
	  actionThread.start(); }        
	else if (SHOW_UNREACHABLE_STR.equals(arg))
	  showUnreachables();
	else if (MINIMIZE_STR.equals(arg)) {
	  actionThread = new DialogThread(this, MINIMIZE);
	  actionThread.start(); } 
	else if ("About...".equals(arg))
	  about();
	else if (HLP_STATE_STR.equals(arg))
	  help(HelpDialog.STATE + typeForHelp());
	else if (HLP_TRANSITION_STR.equals(arg))
	  help(HelpDialog.TRANSITION + typeForHelp());
	else if (HLP_UPPERWINDOW_STR.equals(arg))
	  help(HelpDialog.UPPERWINDOW + typeForHelp());
	else if (HLP_TRACEWINDOW_STR.equals(arg))
	  help(HelpDialog.TRACEWINDOW + typeForHelp());
	else if (HLP_OVERVIEW_STR.equals(arg))
	  help(HelpDialog.OVERVIEW + typeForHelp());
	else if (HLP_MENUBAR_STR.equals(arg))
	  help(HelpDialog.MENUBAR + typeForHelp());
	else if (HLP_NFATODFA_STR.equals(arg))
	  help(HelpDialog.NFATODFA + typeForHelp());
	else if (HLP_MINIMIZE_STR.equals(arg))
	  help(HelpDialog.MINIMIZE + typeForHelp());
	else if ("Load example".equals(arg))
	  loadExample();
	else if ("tabular".equals(arg)) {
	  actionThread = new DialogThread(this, PRINT);
	  actionThread.start(); }
	else if ("postscript".equals(arg)) 
	  printPostscript();
	else if(TO_DFA_STR.equals(arg)) {
	  actionThread = new DialogThread(this, TO_DFA);
	  actionThread.start();} 
	else if ( SHOW_STR.equals(arg))
	  showDfa();
	else if (EXPAND_STATE_STR.equals(arg))
	  expandState();
	else if (DONE_STR.equals(arg))
	  checkDfaDone();
	else if(SHOW_GRAM.equals(arg))
	  showGrammar();
        else {
	  OKDialog dg = new OKDialog("This feature is not implemented", this, "Sorry...");
	  dg.show(); 
	}
    } else if (e.getSource() instanceof Button) {
      String arg = ((Button)(e.getSource())).getLabel();
      if(DONE_MINIMIZE_STR.equals(arg)) {
	if(((TreeMachine) a).checkMinimizeDone()) {
	  ((MainMenu) mainmenu).actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, DONE_MINIMIZE_STR));
	}
      } else if (SHOW_ALL_MINIMIZE_STR.equals(arg)) {
	((TreeMachine)a).showAllMinimize();
      } else {
	OKDialog dg = new OKDialog("This feature is not implemented", this, "Sorry...");
	dg.show(); 
	}
    }
    
    repaint(); 
  }	// end of method action.
  
  /**
   * Relabels the states by calling relabel from the Desktop.
   */
  public void relabel() {
    a.d.relabel();
    a.d.repaint();
    updateStatus("Relabel completed.");
  }

  /** 
    * Pops up a menu and prints the window to a postscript file 
    */
  public void printPostscript() {
    
    try {
      PrintJob pj = this.getToolkit().getPrintJob(this, "Print Screen", null);
      Graphics pg = pj.getGraphics();
      this.printComponents(pg);
      pg.dispose();
      pj.end();
      updateStatus("Print completed.");
    } catch (Exception e) {
      updateStatus("");
      if(e.getMessage() !=null) {     //cancel button 
	OKDialog dg = new OKDialog("Could not print current machine.\n" + e.getMessage(), 
				   this, strType() + " Print Error");
	dg.show();
      }
    }
  }
  
  /**
   * Shows the print dialog and prints the machine in tabular form.
   */
  public void print() {
    PrintDialog pd = new PrintDialog(this);
    String header = "Machine type: " + strType() + "\n\n";
    String[] env = new String[1];
    env[0] = "LD_LIBRARY_PATH=" +System.getProperty("LD_LIBRARY_PATH");
    pd.pack();
    Thread dt = Thread.currentThread();
    pd.setDialogThread(dt);
    pd.show();

    if (pd.getPushedButton().equals(ThreadDialog.NOT_YET))
      dt.suspend();
  
    if ("Print".equals(pd.getPushedButton())) {
      if (pd.printFormat == PrintDialog.TABULAR) {
        try {
          updateStatus("Printing...");
          String filename = (pd.destination == PrintDialog.PRINTER) ? 
          	"tabular.doc" : pd.getFileName();
          if (!filename.startsWith(MainMenu.fileSeparator))
            filename = MainMenu.workingDirectory + filename;
          PrintWriter out = new PrintWriter(new FileOutputStream(filename));
          out.print(header + a.d.toTabular());
          out.close();
          if (pd.destination == PrintDialog.PRINTER) {
            if (MainMenu.osName.equals(Params._sys_macos))
              throw (new Exception("Not implemented for this platform."));
            Process p = Runtime.getRuntime().exec(pd.getCommand() 
			+ " " + filename);
            int code = p.waitFor();
            if (code != 0) 
              throw (new Exception("Printing program error. Exit code "+code));
          }
          if (pd.destination == PrintDialog.PRINTER) {
            String deleteCommand = "";
            if (MainMenu.osName.equals(Params._sys_win95))
              deleteCommand = "del tabular.doc";
            else
              deleteCommand = "/usr/bin/rm tabular.doc";
            Process p = Runtime.getRuntime().exec(deleteCommand);
          }            
          updateStatus("Print completed.");
        } catch (InterruptedException e) {
        } catch (Exception e) {
          updateStatus("");
          OKDialog dg = new OKDialog("Could not print current machine.\n"
                       + e.getMessage(), this, strType() + " Print Error");
          dg.show();
        }
      }
    }	// end of if ("Print".equals...
  }

  /**
   * Opens a dialog and focuses the nondeterministic states.
   * These are found by calling showNondets from Machine.
   * @see	flap.Machine
   */
  public void  showNondets() {
    String error = a.checkMachine();
    if (error == null)
      if (a.showNondets()) {
        a.d.repaint();
        OKDialog dg = new OKDialog("The nondeterministic states\nare highlighted.",
        	this, strType() + " Message");
        Thread dt = Thread.currentThread();
        dg.setDialogThread(dt);
        dg.show();

        if (dg.getPushedButton().equals(ThreadDialog.NOT_YET))
          dt.suspend();   
        a.d.unfocusAll();
        a.d.repaint();
      } else {
        OKDialog dg = new OKDialog(AUTOMATON_IS_DETER_STR, this,
        	strType() + " Message");
        dg.show();
      }
    else {
      OKDialog dg = new OKDialog(error, this, strType() + " Error");
      dg.show();
    }    
  }

  /**
    * Checks if the machine is deterministic or has any errors and returns an appropriate String.
    */
  public String checkDeterministic() {
      String error = a.checkMachine();
      if (error != null )
	return error;
      if (a.showNondets() == false)
	return AUTOMATON_IS_DETER_STR;
       a.d.unfocusAll();
       a.d.repaint();
      return AUTOMATON_IS_NON_DETER_STR;
    }

  /**
    * sets the FSA to be a DFA builder
    */
  public void setToDfa(Environment NFA) {
    Desktop temp;

    setEnvSource(NFA);
    NFA.setEnvTarget(this);
    temp = new Desktop(Machine.FSA);
    dfa = new FSA(temp);
    dfa.buildDfa((FSA) NFA.a);
    updateStatus("Mail any bugs or comments to: rodger@cs.duke.edu");
    nfaToDfa.setEnabled(false);
    isDone.setEnabled(true);
    show.setEnabled(true);
    expand.setEnabled(true);
    showNameLabels.setEnabled(true);
    showNameLabels.setState(true);
    d.showDfaLabels = true;
    dfa.d.showDfaLabels = true;
  }

  public void setToMinimizeFinal(Environment theSource) {
    Desktop temp;
    int trap;
    oldTree = theSource;
    setEnvSource(theSource.envSource);
    theSource.envSource.setEnvTarget(this);
    theSource.setEnabled(false);
    theSource.setEnvSource(null);
    temp = new Desktop(Machine.FSA);
    dfa = new FSA(temp);
    ((FSA)a).putMinimizedStates((TreeMachine) theSource.a);
    dfa.buildMinimized((TreeMachine) theSource.a, (FSA) envSource.a);
    updateStatus("Mail any bugs or comments to : rodger@cs.duke.edu");
    minimize.setEnabled(false);
    minIsDone.setEnabled(true);
    minShow.setEnabled(true);
    minExpand.setEnabled(true);
    showNameLabels.setEnabled(true);
    showNameLabels.setState(true);
    d.showDfaLabels = true;
    dfa.d.showDfaLabels = true;
    validate();
  }

  /**
    * sets the Environment to be a Minimize Tree environment
    */
  public void setToMinimizeTree(Environment theSource) {
    setEnvSource(theSource);
    theSource.setEnvTarget(this);
    theSource.d.showDfaLabels = false;
    theSource.d.repaint();
    d.showDfaLabels = true;
    runmenu.setEnabled(false);
    optionsmenu.setEnabled(false);
    helpmenu.setEnabled(false);
    newItem.setEnabled(false);
    openItem.setEnabled(false);
    saveItem.setEnabled(false);
    saveAsItem.setEnabled(false);
    tabularItem.setEnabled(false);
    isMinimizeDone.setVisible(true);
    showAllMinimize.setVisible(true);
    inputField.setVisible(false);
    inputName.setVisible(false);
    int trap = ((TreeMachine) a).startTree(envSource.a);
    trap = (trap != -1) ? (updateStatus("State q" + trap + " is the implicit trap state.")) : (updateStatus("Mail any bugs or comments to : rodger@cs.duke.edu"));
    validate();
  }

  /**
    * Sets the source Environment that generated this Environment
    */
  public void setEnvSource(Environment source) {
    envSource = source;
  }

  /**
    * Sets the Environment that is generated by this environment
    */
  public void setEnvTarget(Environment target) {
    envTarget = target;
    setEnabled(target == null);
  }

  public Environment getEnvSource() {
    return envSource;
  }

  /**
    * Show the regular grammar corresponding to this FSA 
    */
  public void showGrammar() {
    
    if(d.initialState == null) {
      String erro = "The machine has no initial state";
      OKDialog dg = new OKDialog(erro, this, "Grammar error.");
      dg.show(); 
      return;
    }

    FSAToGrammar.displayGrammar(d);
  }


  /**
    * Show the answer to the DFA convertion problem
    */
  public void showDfa() {
    if (dfa == null)
      return;

    Machine temp = a;
    a = new FSA(dfa);
    d = a.d;
    this.add("Center", d);
    temp.d.setVisible(false);
    d.setVisible(true);
    validate();
    ((FSA) a).placeAllStates(temp, (FSA) (envSource.a));
    d.repaint();
  }

  /**
    * Handles the checking to see if DFA and NFA are identical
    */
  public void checkDfaDone() {
    String erro;

    if (dfa == null)
      return;

    d.unmarkAllStates();
    dfa.d.unmarkAllStates();
    d.unselectTransitions();

    erro = checkDeterministic();
    if(erro != AUTOMATON_IS_DETER_STR) {  
      a.showNondets();
      if( erro.equals(AUTOMATON_IS_NON_DETER_STR))
	erro = "The highlighted state(s) are nondeterministic";
      OKDialog dg = new OKDialog(erro, this, "Automaton check.");
      dg.show(); 
    } else if(checkMultiLetterLabels()) {
      erro = "Your automaton should have no multiple \n letter labels after being converted.";
      OKDialog dg = new OKDialog(erro, this, "Automaton check.");
      dg.show(); 
    } else {
      erro = ((FSA) a).checkDfaDone(dfa, envSource.a);
      d.repaint();
      OKDialog dg = new OKDialog(erro, this, "Automaton check.");
      dg.show(); 
    }
    d.unfocusAll();
    d.repaint();
    d.selectedState = null;
  }

  /**
    * Solves only the selected State in the NFA to DFA window
    */
  public void expandState() {
    String answer = ((FSA) a).expandState(dfa, (FSA) (envSource.a));
    if(!answer.equals(DFA_IS_CORRECT_STR)) {
      OKDialog dg = new OKDialog(answer, this, "Expand State.");
      dg.show();
    }
    d.repaint();
  }
  
  /**
    * Check if any of the transitions have more than one letter.
    */
  public boolean checkMultiLetterLabels() {
    LabelSeparator breaker = new LabelSeparator((FSA) a);
    return breaker.checkMultiLetterLabels();
  }

  /**
    * Breaks up all multiple letter labels on the desktop into several states.
    */
  public void breakupMultipleLabels() {
    LabelSeparator breaker = new LabelSeparator((FSA) a);
    breaker.breakupMultipleLabels();
  }

  /**
    * minimizes the number of states in the FSA
    */
  public void minimize() {
    String error = null;

    if(d.initialState == null)
      error = "The machine has no initial state.";
    else if(!a.showUnreachables().equals(FSA.NO_UNREACHABLES_STR)) { 
      error = "You must remove all unreachable \n states first. (you can use the \n\"Show unreachable\" button for help)";
      d.unfocusAll();
    } else if( checkDeterministic().equals(AUTOMATON_IS_NON_DETER_STR)) {
      a.showNondets();
      error = "The highlighted states are nondeterministic. \n Nondeterministic machines cannot be \n minimized";
    } else if (!checkDeterministic().equals(AUTOMATON_IS_DETER_STR)) 
      error = checkDeterministic();
    if(error != null) {
      OKDialog dg = new OKDialog(error, this, "Minimize.");
      d.repaint();
      dg.show();
      d.unfocusAll();
      d.repaint();
      return;
    }

    if(checkMultiLetterLabels() == true ) {
      TextInDialog dg = new TextInDialog( MainMenu.LABEL_WARNING_STR, MainMenu.OK_CANCEL_STR, 
					  this, strType() + " Warning", false);
      Thread dt = Thread.currentThread();
      dg.setDialogThread(dt);
      dg.show();
      if (dg.getPushedButton().equals(ThreadDialog.NOT_YET))
	dt.suspend();   
      String pushed = dg.getPushedButton();
      if (pushed.equals(MainMenu.OK_STR)) 
	breakupMultipleLabels();
      return;
    } 
 
    ((MainMenu) mainmenu).actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, 
							    MINIMIZE_STR));
  }


  /**
   * Shows the states which are unreachable from
   * the initial state.
   */
  public void showUnreachables() {
     OKDialog dg = new OKDialog(a.showUnreachables(), this, "Unreachable States.");
     d.repaint();
     dg.show();
     d.unfocusAll();
     d.repaint();
  }

  /**
   * Removes the current desktop and opens a brand new one.
   */
  void newone() {
    if (!testSavedAndProceed())
      return;
    remove(d);
    d = new Desktop(type);
    switch (type) {
      case Machine.FSA:
        a = new FSA(d);
	d.showDfaLabels = showNameLabels.getState();
        break;
      case Machine.PDA:
        a = new PDA(d);
        break;
     case Machine.TM1:
        a = new TM1(d);
        break;
     case Machine.TM2:
        a = new TM2(d);
        inputField2.setText("");
        break;
    }
    inputField.setText("");
    add("Center", d);
    fileName = NO_FILE_NAME;
    doLayout();
    updateStatus();
  }

  /**
   * Shows the help dialog.
   */
  public void help(int helpOnWhat) {
    HelpDialog hd = new HelpDialog(helpOnWhat, this);
    hd.pack();
    hd.show();
  }

  /**
   * Shows the about dialog.
   */
  public void about() {
    OKDialog dg = new OKDialog("\nJFLAP v. 2.0, July 31, 1997\n (c) 1997 Susan H. Rodger and\n Eric Gramond. \n ---------------------------- \n Flap v. 1.0 July 15 1996\n (c) 1996 Susan H. Rodger,\nMagda Procopiuc &\nOctavian Procopiuc.", this, "About jflap");
    dg.setSize(300, 225);
    dg.show();
  }

  /**
   * Shows a FileDialog and tries to open the selected file.
   */
  public boolean open() {
    boolean answer = false;

    if (!testSavedAndProceed())
      return false;
    FileDialog fd = new FileDialog(this, "Open " + strType(), FileDialog.LOAD);
    fd.setBackground(Params._dg_backcolor);
    if (currentDir != null)
      fd.setDirectory(currentDir);
    fd.show();
    String strFile = fd.getFile();
    String strDir = fd.getDirectory();
    if (strDir == null || strFile == null)
      return false;
    if (strFile.endsWith(".*.*"))  // ugly bug in JDK 1.0.2 for win32.
      strFile = strFile.substring(0, strFile.length() - 4);
    if (!strDir.endsWith(MainMenu.fileSeparator))
      strDir += MainMenu.fileSeparator;
    currentDir = strDir;
    String aName = strDir + strFile;
    if (!aName.endsWith("." + strType()))
      aName += "." + strType();
    try {
      open(aName);
      answer = true;
    }
    catch (FileNotFoundException e) {
      OKDialog dg = new OKDialog("Could not find file\n"+"'" + aName + "'", 
		this, strType() + " Open Error");
      dg.show();
    }
    catch (EOFException e) {
      OKDialog dg = new OKDialog("Unexpected end of file.", this, 
		strType() + " Open Error");
      dg.show();
    }
    catch (NumberFormatException e) {
      OKDialog dg = new OKDialog("Numbers not found where\nthey were needed...",
		this, strType() + " Open Error");
      dg.show();
    }
    catch (IOException e) {
      OKDialog dg = new OKDialog("An I/O error occured.", this, 
		strType() + " Open Error");
      dg.show();
    }
    catch (BadInputException e) {
      OKDialog dg = new OKDialog("File corrupted" + e.getMessage(),
		this, strType() + " Open Error");
      dg.show();
    }
    return answer;
  }

  /**
   * Tries to open the given file.
   * @exception flap.BadInputException  if the input is badly formed
   * @exception IOException if the file cannot be read correctly
   * @exception NumberFormatException if the file contains a string isntead of a number
   * @exception EOFException if the file ends too early
   */
  public void open(String aName) throws BadInputException, IOException, NumberFormatException, EOFException {
    boolean	answer = false;
    FileRead	fr = new FileRead();

    if (!aName.endsWith("." + strType()))
      aName += "." + strType();
    fr.read(aName);
    if (fr.a != null) {	// just in case...
      a = fr.a;
      remove(d);
      d = a.d;
      add("Center", d);
    }
    inputField.setText("");
    doLayout();
    fileName = aName;
    updateStatus();
  }

  /**
   * Loads an example machine from a URL given in Params.
   * Applies only when isApplet is true.
   */
  public void loadExample() {
    String exampleurl = Params._e_exampleurl + strType();
    try {
      URL url = new URL(exampleurl);
      InputStream is = url.openStream();
      FileRead fr = new FileRead();
      fr.setStream(is);
      fr.parseStream();
      if (fr.a != null) {	// just in case...
        a = fr.a;
        remove(d);
        d = a.d;
        add("Center", d);
        updateStatus("Example loaded from " + exampleurl);
      }
      inputField.setText("");
      doLayout();
    } catch (Exception e) {
      OKDialog dg = new OKDialog("Error while loading or parsing.\n" + e.getMessage(), this, "Load Example Error");
      dg.show();
    }         
  }
    
  /**
   * Tries to save the current machine.
   */
  public boolean save() {
    if (NO_FILE_NAME.equals(fileName))
      return saveAs();
    else
      return save(fileName);
  }

  /**
   * Opens a FileDialog and tries to save the current machine
   * in the specified file.
   */
  public boolean saveAs() {
    FileDialog fd = new FileDialog(this, "Save " + strType(), FileDialog.SAVE);
    if (currentDir != null)
      fd.setDirectory(currentDir);
    fd.setBackground(Params._dg_backcolor);
    fd.show();
    String strFile = fd.getFile();
    String strDir = fd.getDirectory();
    if (strDir == null || strFile == null)
      return false;
    if (strFile.endsWith(".*.*"))  // ugly bug in JDK 1.0.2 for win32.
      strFile = strFile.substring(0, strFile.length() - 4);
    if (!strDir.endsWith(MainMenu.fileSeparator))
      strDir += MainMenu.fileSeparator;
    currentDir = strDir;
    String aName = strDir + strFile;
    return save(aName);
  }

  /**
   * Tries to save the current machine in the file given
   * by the name.
   */
  public boolean save(String aName) {
    boolean answer = false;
    FileWrite fw = new FileWrite(a);

    try {
      if (aName.trim().endsWith(MainMenu.fileSeparator))
        throw (new Exception("Not a valid file name.\n"));
      if (!aName.endsWith("."+strType()))
        aName += "." + strType();
      fw.write(aName);
      fileName = aName;
      a.d.modified = false;
      updateStatus("File saved");
      answer = true;
    }
    catch (IOException e) {
      OKDialog dg = new OKDialog("I/O Error.\nFile not saved.\n"
		+ e.getMessage(), this, strType() + " Save Error");
      dg.show();
    }
    catch (NoSuchElementException e) {
      OKDialog dg = new OKDialog("Tokenizer Error.\nFile not saved.\n"
		+ e.getMessage(), this, strType() + " Save Error");
      dg.show();
    }
    catch (Exception e) {
      OKDialog dg = new OKDialog("File Error.\nFile not saved.\n"
		+ e.getMessage(), this, strType() + " Save Error");
      dg.show();
    }
    return answer;
  }	// end of method save(String).

  public void setEnabled(boolean b) {
    super.setEnabled(b);
    runmenu.setEnabled(b);
    filemenu.setEnabled(b);
    optionsmenu.setEnabled(b);
  }

  public void close() {
    if (!testSavedAndProceed())
      return;
    ((MainMenu) mainmenu).actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, 
							  CLOSE_STR));
  }
 
  public void to_dfa() {
    String determ = checkDeterministic();

    if(!determ.equals(AUTOMATON_IS_NON_DETER_STR)) {
      OKDialog dg = new OKDialog(determ, this, strType() + " Message");
      dg.show();
      return;
    }
    if(checkMultiLetterLabels() == true ) {
      TextInDialog dg = new TextInDialog( MainMenu.LABEL_WARNING_STR, MainMenu.OK_CANCEL_STR, 
					  this, strType() + " Warning", false);
      Thread dt = Thread.currentThread();
      dg.setDialogThread(dt);
      dg.show();
      if (dg.getPushedButton().equals(ThreadDialog.NOT_YET))
	dt.suspend();   
      String pushed = dg.getPushedButton();
      if (pushed.equals(MainMenu.OK_STR)) 
	breakupMultipleLabels();
      return;
    } 
    ((MainMenu) mainmenu).actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, 
							  TO_DFA_STR));
  }

  /**
   * Opens the Step Run Window.
   */
  public void stepRun() {
    String	error = a.checkMachine();
    String	input = inputField.getText();

    if (error == null) {
      setEnabled(false);
      if (type == Machine.TM2) {
        input += " | " + inputField2.getText();
      }
      a.setInput(input);
      a.init();
      d.changeAspect(true);
      UpperWindow uw = new UpperWindow(strType() + " Step Run", this);
      uw.pack();
      uw.show();
    } else {
      OKDialog dg = new OKDialog(error, this, strType() + " Error");
      dg.show();
    }
  }	// end of method stepRun.

  /**
   * Does a fast run and shows the result.
   * Gives the opportunity to see the path
   * that lead to acceptance.
   */
  public void fastRun() {
    String	error = a.checkMachine();
    String	input = inputField.getText();

    if (error == null) {
      if (type == Machine.TM2) {
        input += " | " + inputField2.getText();
      }
      a.setInput(input);
      a.init();
      a.d.initialState.ss = State.NORMAL;
      String[] str = {"Yes", "No"};
      done = false;
      while (!done) {
        done = true;
	TextInDialog dg;
	Thread dt;
        switch (a.fastRun()) {
         
          case Machine.NO_FINAL_STATE_MSG:
              dg = new OKDialog("Input Rejected.\n"
              + "(machine has no final state)", this, strType() + " Fast Run");
              dg.show();
              break;
          case Machine.MAXNCONFIGS_MSG:
            if (a.maxNConfigs < a.capacity) {
              dg = new TextInDialog("More than "+a.maxNConfigs
              	+" configurations\n have been generated and the input is\n"
              	+"not accepted yet. Continue?", str, this, strType()
		+ " Fast Run Warning");  
              dt = Thread.currentThread();
              dg.setDialogThread(dt);
              dg.show();
              if (dg.thePushedButton.equals(ThreadDialog.NOT_YET))
                dt.suspend();

              if (dg.getPushedButton().equals("Yes")) {
                a.maxNConfigs *= 2;
                done = false;  
              }  
            } else {
              dg = new OKDialog("More than "+a.capacity
		+" configurations\n have been generated\n."
		+" Unable to continue processing", this, strType()
		+ " Fast Run Error");
              dg.show();
            }
            break; 
          case Machine.ACCEPTED_MSG:
            Configuration c = a.acceptingConfig;
            int pathLength = 1;
            while (c.father != null) {
              c = c.father;
              pathLength++;
            }
            dg = new TextInDialog("Input Accepted!\n"+a.nConfigs
            	+" configurations have been generated.\n"
            	+" The acceptance path has length "
		+pathLength+"\n Input was accepted in state q"
		+a.acceptingConfig.theState.id+"\nWant to see the acceptance path?",
		 str, this, strType() + " Fast Run");  
            dt = Thread.currentThread();
            dg.setDialogThread(dt);
            dg.show();

            if (dg.thePushedButton.equals(ThreadDialog.NOT_YET))
              dt.suspend(); 
            if (dg.getPushedButton().equals("Yes")) {
              TraceWindow tw = new TraceWindow(a, a.acceptingConfig, this,
              	 strType() + " Accepting Path");
              tw.show();              
            }   
            break;
          case Machine.NO_MORE_CONFIGS_MSG:
            dg = new OKDialog("Input rejected", this, strType() + " Fast Run");
            dg.show();
            break;
        }  // end switch
      } // end while (!done)
    } else {
      OKDialog dg = new OKDialog(error, this, "Error");
      dg.show();
    }
    a.d.repaint();
  }	// end of method fastRun.

  /**
   * Updates the title and the status line.
   */
  public void updateStatus() {
    String typeInTitle = (type == Machine.FSA) ? ( 
      (envSource != null) ? (oldTree == null) ? "NFA to DFA builder" : 
           "Minimize Builder" : "Finite State Automaton") :
      ((type == Machine.PDA) ? "Pushdown Automaton" : 
       ((type == Machine.TM1) ? "One-Tape Turing Machine" :
	((type == Machine.TM2) ? "Two-Tape Turing Machine" : "Minimize Tree builder")));

    String strModified = (a.d.modified) ? "(modified)":"";
    setTitle(typeInTitle + ": " + fileName + " " + strModified);
    statusLine.setText("");
  }

  /**
   * Updates the title and the status line.
   * @param message	the new status line.
   */
  public int updateStatus(String message) {
    updateStatus();
    statusLine.setText(message);
    return 0;
  }

  /**
   * Tests if the current machine is modified,
   * asks to save it if it is not modified.
   * @return	false if the user pushed the cancel button, true otherwise.
   */
  public boolean testSavedAndProceed() {
    boolean answer = true;
    if (isApplet)
      answer = true;
    else if (a.d.modified) {
      TextInDialog dg = new TextInDialog("Current machine is not saved", SAVE_BUTTONS, this, strType() + " Warning");
      Thread dt = Thread.currentThread();
      dg.setDialogThread(dt);
      dg.show();

      if (dg.getPushedButton().equals(ThreadDialog.NOT_YET))
        dt.suspend();   
      String pushed = dg.getPushedButton();
      if (pushed.equals(SAVE_STRING)) {
        if (!save())
          answer = false;
      }
      else if (pushed.equals(CANCEL_STRING))
        answer = false;
 
    }
    return answer;
  }	// end of method testSavedAndProceed.

  /**
   * Returns a number representing the current type.
   * Used in HelpDialog. This is ugly.
   */
  protected int typeForHelp() {
    int answer = -1;

    switch (type) {
      case Machine.FSA:
        answer = 0;
        break;
      case Machine.PDA:
        answer = 1;
        break;
      case Machine.TM1:
        answer = 2;
        break;
      case Machine.TM2:
        answer = 3;
        break;
    }
    return answer;
  }

  /**
   * Returns a string representing the type of the
   * current machine.
   */
  protected String strType() {
    String answer = "";

    switch (type) {
      case Machine.FSA:
        answer = "FA";
        break;
      case Machine.PDA:
        answer = "PDA";
        break;
      case Machine.TM1:
        answer = "TM";
        break;
      case Machine.TM2:
        answer = "TTM";
        break;
    }
    return answer;
  }	// end of method strType.

  /**
   * Should be removed when the bug regarding non-modal
   * dialogs is fixed.
   */
  public void threadAction(int id) {
    
    switch (id) {
    case OPEN:
      open();
      break; 
    case PRINT:
      print();
      break;
    case CLOSE:
      close();
      break;
    case NEW:
      newone();
      break;
    case FASTRUN:
      fastRun();
      break;
    case SHOWNONDETS:
      showNondets();
      break;
    case TO_DFA:
      to_dfa();
      break;
    case MINIMIZE:
      minimize();
      break;
    }
  }  // end of method threadAction

}	// end of class Environment.




