/*---------------------------------------------------------------------------
File:         			Desktop.java
Package:                        JFLAP Version 1.0
Author:				Magda & Octavian Procopiuc V1.0 07/15/96
                                
Description of Contents:	Contains class Desktop.
				
--------------------------------------------------------------------------*/

/* 
 * Susan H. Rodger, Magda Procopiuc, Octavian Procopiuc, Eric Gramond
 * Computer Science Department
 * Duke University
 * June 1997
 * Supported by National Science Foundation DUE-9596002 and DUE-9555084.
 *
 * Copyright (c) 1997
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the author.  The name of the author may not be used to
 * endorse or promote products derived from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

package flap;

import	java.awt.*;
import  java.awt.event.*;
import	java.util.*;


/**
 * This is the canvas where the machines are built and displayed.
 * It is the main class involved in the interface part of 
 * the different
 * machines. The lists of transitions and states are kept here, 
 * and the actual painting of the transitions and states is performed
 * here also.
 *
 * @see		flap.State
 * @see		flap.Transition
 * @author	Magda & Octavian Procopiuc
 * @version	1.0, 15 July 1996
 */ 
public class Desktop extends Canvas implements MouseMotionListener, KeyListener, MouseListener {
    
  /**
   * The start state of the machine.
   */
  public State		initialState = null;
  /**
    * The currently selected transition.
    */
  public Transition	selectedTransition; 
  /**
    * The currently selected dfa State (for label)
    */
  public State          selectedState;
  /**
    * The list of states currently on the desktop.
    * The machine will use them from here, too.
    */
  public Vector		theStates;
  /**
   * The list of transitions currently on the desktop.
   * The machine will use them from here, too.
   */
  public Vector		theTransitions;
  /**
   * Becomes true when the current machine is modified in any way.
   */
  public boolean	modified = false;
  /**
   * The type of the machine currently on the desktop.
   * This enables us to draw any type of machine using this class.
   */
  public int		type;
  /**
    * whether to show State labels or not.
    */
  public boolean showDfaLabels = false;
  /**
   * When the user sets the auto relabel option,
   * this flag becomes true, and the states are
   * automatically relabeled each time one of them is removed.
   */
  boolean	autoRelabel = false;
  /**
   * The state label counter.
   */ 
  int		currentId = 0;
  /**
   * The x distance between the mouse pointer
   * and the center of the state circle, when
   * dragging a state.
   */
  protected int		x_offset = 0;
  /**
   * The x distance between the mouse pointer
   * and the center of the state circle, when
   * dragging a state.
   */
  protected int		y_offset = 0;
  protected Color	backcolor = Params._d_backcolor;
  protected PopupMenu	menu;
  protected boolean	menuOn = false;
  /**
   * used to retain the beginning state when dragging.
   */
  protected State	firstState = null;
  /**
   * used when dragging.
   */
  protected Transition	firstTransition = null;
  protected Point	drag = null;
  protected int		draggingWhat;
  /**
   * The number of sublabels in a label.
   */
  protected int		fieldCounter = 1;
  /**
   * The separators between the sublabels in a label.
   */
  protected String[]	separator = new String[6];

  /**
   * What are we currently dragging.
   */
  final static int DG_NOTHING	= 0;
  final static int DG_STATE	= 1;
  final static int DG_TRANS	= 2;
  final static int DG_LABEL	= 3;
  /**
   * Used for writing the current machine in the tabular form.
   * @see	toTabular()
   */
  final static String EMPTY_STR = "-----";


  private final static int shiftAndButton = Event.SHIFT_MASK | InputEvent.BUTTON1_MASK;
  private final static int altAndButton = Event.ALT_MASK | InputEvent.BUTTON1_MASK;
  
  /**
   * the offscreen image.
   */
  private Image		offscr = null;
  /**
   * the graphics context of the offscreen image.
   */
  private Graphics	offg = null;

  private int	off1 = 11;
  private int	off2 = 6;
  private int 	count = 0; 

  /**
   * distance factor between transitions
   */
  private double df = 1.5;

  /**
    * value used in compareTo to initialise unique checkmarks
    */
  public int currentCheckMark = 0;

  /**
   * Builds a new Desktop with the specified type.
   * @param type	the type of the machine represented on the desktop.
   */
  public Desktop(int type) {
    this.type = type;

  /**
   * Overrides processKeyEvent so that tabs can be
   * consumed by Desktop
   */
    this.enableEvents(AWTEvent.KEY_EVENT_MASK);
    
    this.addMouseMotionListener(this);
    this.addMouseListener(this);
    this.addKeyListener(this);
    
    theStates = new Vector(10, 1);
    theTransitions = new Vector(30, 1);
    menu = new Popup(this);
    setupSeparators();
  }

  /** 
    * copy constructor. makes a deep copy.
    * Precondition: all State ids are unique.
    */
  public Desktop(Desktop aDesktop) {
    this(aDesktop.type);

    Enumeration e;
    this.currentId = aDesktop.currentId;
    this.autoRelabel = aDesktop.autoRelabel;
    this.showDfaLabels = aDesktop.showDfaLabels;

    int idFrom, idTo;
    State stateFrom, stateTo;
    Transition aTransition;

    for( e = aDesktop.theStates.elements(); e.hasMoreElements(); ) 
      theStates.addElement(new State( (State) e.nextElement()));
    
    for( e = aDesktop.theTransitions.elements(); e.hasMoreElements(); ) {
      aTransition = (Transition) e.nextElement();
      idFrom = aTransition.from.id;
      idTo = aTransition.to.id;
      stateFrom = getStateWithId(idFrom);
      stateTo = getStateWithId(idTo);
      this.addTransition(stateFrom, stateTo).label = new String(aTransition.label);
    }
    
    if(aDesktop.initialState != null)
      this.initialState = getStateWithId(aDesktop.initialState.id);
  }


  /**
    * sets up the separators for transition labels for different types of automatons
    */
  void setupSeparators() {
 switch (type) {
      case Machine.FSA:
        fieldCounter = 1;
        break;
      case Machine.PDA:
        separator[0] = ",";
        separator[1] = ";";
        fieldCounter = 3;
        break;
      case Machine.TM1:
        separator[0] = ";";
        separator[1] = ",";
        fieldCounter = 3;
        break;
      case Machine.TM2:
        separator[0] = ";";
        separator[1] = ",";
        separator[2] = "|";
        separator[3] = ";";
        separator[4] = ",";
        fieldCounter = 6;
        break;
    }
    separator[fieldCounter - 1] = "";
  }

  /**
    *  Overrides Component.processKeyEvent so that tabs are 
    *  consumed here in the canvas instead of letting the
    *  canvas go out of focus and highlighting another component
    */
  protected void processKeyEvent(KeyEvent e) {
   
   if (e.getID() == KeyEvent.KEY_PRESSED){
      int key;
      if (!e.isActionKey()) // if is a character key
	key = e.getKeyChar();
      else key = e.getKeyCode();
      requestFocus();

      if(selectedState != null)
	modified = selectedState.processKey(key, e) || modified;
      else if(selectedTransition != null)
	modified = selectedTransition.processKey(key, e) || modified;
      repaint();	 
      e.consume(); // must consume so that a tab will not take the Desktop out of focus
   }
  }

  public void keyPressed(KeyEvent e) 
  {}
  public void keyTyped(KeyEvent e) 
  {}
  public void keyReleased(KeyEvent e)
  {}

  public void mouseClicked(MouseEvent e) 
  {}
  public void mouseEntered(MouseEvent e) 
  {}
  public void mouseExited(MouseEvent e) 
  {}
  public void mousePressed(MouseEvent e){

    State	s;
    Transition	t;
    
    int x = e.getX();
    int y = e.getY();

    requestFocus();

    //why was this done with a switch statement?? It's a bitvector!!
    switch (e.getModifiers()) {
    case Event.ALT_MASK:	// middle button down.
    case Event.SHIFT_MASK:
    case shiftAndButton:
    case altAndButton:
       if ((s = isInState(x,y)) != null) {
          draggingWhat = Desktop.DG_STATE;
          firstState = s;
          x_offset = x - s.p.x;
          y_offset = y - s.p.y;
          modified = true;
        }
        else if (isGoodLocation(x, y)) {
          s = addState(x, y);
          if (theStates.size() == 1)
            initialState = s;
          draggingWhat = Desktop.DG_STATE;
          firstState = s;
          x_offset = 0;
          y_offset = 0;
          modified = true;
        }
        else {
          draggingWhat = Desktop.DG_NOTHING;
        }
        break;
      	// end of case ALT_MASK...

    case 0:			// left button down.
    case InputEvent.BUTTON1_MASK:
        if ((t = isInTransition(x,y)) != null) {
          draggingWhat = Desktop.DG_LABEL;
          firstTransition = t;
          Point pp = locateLabel(t);
          x_offset = x - pp.x;
          y_offset = y - pp.y;
          selectedTransition = t;
	  selectedState = null;
	  t.tokenizeLabel();
	} else if ((s = isInStateDfaLabel(x,y)) != null) {
	  draggingWhat = Desktop.DG_NOTHING;
	  firstState = s;
	  selectedTransition = null;
	  selectedState = s;   
        } else if ((s = isInState(x,y)) != null) {
	  draggingWhat = Desktop.DG_TRANS;
          firstState = s;
          drag = new Point(x, y);
        } else {
          draggingWhat = Desktop.DG_NOTHING;
        }
        break;
        // end of case 0

      case Event.META_MASK:	// right button down.
      case Event.CTRL_MASK:
        menuOn = true;
        menu.move(x, y);
        menu.invalidateAll();
        if ((s = isInState(x, y)) != null) 
          menu.setState(s);
        else if ((t = isInTransition(x, y)) != null)
          menu.setTransition(t);
        else {}        
        break;
        //end of case InputEvent.MET..
      default:
        break;
    }	// end of switch
    
    repaint();
  }	// end of method MouseDown.
  
  public void mouseMoved(MouseEvent e)
  {}
  
  public void mouseDragged(MouseEvent e) 
  {

    int x = e.getX();
    int y = e.getY();
    switch (e.getModifiers()) {
    case Event.ALT_MASK:	// middle mouse down.
    case Event.SHIFT_MASK:
    case shiftAndButton:
    case altAndButton:
      if (draggingWhat == Desktop.DG_STATE) {
          firstState.move(x - x_offset, y - y_offset, getSize());
        }
        else {}
        break;
        // end of case ALT_MASK...
    case InputEvent.BUTTON1_MASK:			// left mouse down.
    case 0:
	if (draggingWhat == Desktop.DG_TRANS) {
          drag.move(x, y);
        }
        else if (draggingWhat == Desktop.DG_LABEL) {
          if (firstTransition.from == firstTransition.to)
            moveLoop(firstTransition, x - x_offset, y - y_offset);
        }
        else {}
        break;
        // end of case 0
      case Event.META_MASK:
      case Event.CTRL_MASK:
	if (x >= menu.p.x && x <= menu.p.x + menu.width) {
          int idx = (int) Math.floor((y - menu.p.y) / menu.height);
          if (y >= menu.p.y && idx < menu.itemsCounter)
            menu.hasFocus = idx;
          else
            menu.hasFocus = -1;
        } else
          menu.hasFocus = -1;
        break;
    }	// end of switch.

    repaint();
  }	// end of method MouseDrag.

  public void mouseReleased(MouseEvent e){
    State	s;
    Transition	t;

    int x = e.getX();
    int y = e.getY();

    switch (e.getModifiers()) {
      
    case Event.ALT_MASK:	// middle button up.
    case Event.SHIFT_MASK:
    case shiftAndButton:
    case altAndButton:
        if (draggingWhat == Desktop.DG_STATE) {
          draggingWhat = Desktop.DG_NOTHING;
          // what if there's another state here???
          firstState.move(x - x_offset, y - y_offset, getSize());
          x_offset = 0;
          y_offset = 0;
          firstState = null;
        }
        else {}
        break;
        // end of case ALT_MASK...
    case InputEvent.BUTTON1_MASK:			// left button up.
    case 0:
   	if (draggingWhat == Desktop.DG_LABEL) {
          draggingWhat = Desktop.DG_NOTHING;
          if (firstTransition.from == firstTransition.to)
            moveLoop(firstTransition, x - x_offset, y - y_offset);
          x_offset = 0;
          y_offset = 0;
          firstTransition = null;
        } else if (draggingWhat == Desktop.DG_TRANS) {
          if ((s = isInState(x,y)) != null) {
            t = addTransition(firstState, s);
	    selectedTransition = t;
	    selectedState = null;
            if (firstState == s) {	// for loop transitions
              Transition tt = sameStatesTransition(t);
              if (tt == null) {
                t.distance = Params._t_ltheight;	// distance is the height.
              } else
                t.distance = tt.distance;
            }
            modified = true;
          }
          else {}
          drag = null;
          firstState = null;
          draggingWhat = Desktop.DG_NOTHING;
        } else {}
      case Event.META_MASK:
      case Event.CTRL_MASK:
        break;
    }    // end of switch 
    if (menuOn) {	// it's not under switch due to differences in events on
			// different platforms. bugs...
      menuOn = false;
      if (menu.takeAction())
        modified = true;
    }
    repaint();
  }	// end of method MouseUp.


  /**
   * Moves the label of a transition having
   * the same from and to states.
   */
  private void moveLoop(Transition t, int x, int y) {
    boolean done = false;
    Transition tt = null;
    int sz = theTransitions.size();
    int i = 0;
    while (!done && i < sz) {
      tt = (Transition) theTransitions.elementAt(i);
      if ((tt.from == t.from) && (tt.to == t.to))
        done = true;
      else
        i++;
    }
    if (tt == t) {
      for (Enumeration ee=theTransitions.elements(); ee.hasMoreElements();) {
        tt = (Transition) ee.nextElement();
        if (tt.from == firstTransition.from && tt.to == firstTransition.to)
          tt.move(x, y);
      }     
    }
  }	// end of method moveLoop.


  /**
   * Returns the first transition in the list of transitions
   * which has the same from and to states with the given transition t,
   * but is different from it.
   */
  private Transition sameStatesTransition(Transition t) {
    boolean done = false;
    Transition tt = null;
    int sz = theTransitions.size();
    int i = 0;
    while (!done && i < sz) {
      tt = (Transition) theTransitions.elementAt(i);
      if ((tt.from == t.from) && (tt.to == t.to) && (tt != t))
        done = true;
      else
        i++;
    }
    return (done)?tt:null;
  }	// end of method sameStatesTransition

  /**
   * Sets the transition field of a given State to be the list
   * of transitions having that state as the from state.
   * @param s	the state.
   * @param tokenize	whether or not to tokenize a FSA label.
   */
  void setTransitions(State s, boolean tokenize) {
    Transition t;
    StringTokenizer stk;
    String label;

    s.transitions.removeAllElements();
    for (Enumeration et = theTransitions.elements(); et.hasMoreElements(); ) {
      t = (Transition) et.nextElement();     
      if (t.from == s) {
        if (type == Machine.FSA && tokenize) {
          stk = new StringTokenizer(t.label, ",", false);
          while (stk.hasMoreTokens()) {
            label = stk.nextToken().trim();
            s.transitions.addElement(new Transition(s, t.to, label, fieldCounter, separator));
	  }
        } else
          s.transitions.addElement(t);
      }
    }
  }

  public void  paint(Graphics g) {
    
    Transition aTransition;

    Dimension	d = getSize();
    g.setColor(backcolor);
    g.fillRect(0, 0, d.width, d.height);

    g.setColor(Params._d_forecolor);
    g.draw3DRect(0, 0, d.width-1, d.height-1, true);

    for (Enumeration e = theTransitions.elements(); e.hasMoreElements();) {
      ((Transition) e.nextElement()).paintLine(g, selectedTransition);
    }
 
    // although it's very time-consuming to draw everything when dragging,
    // it seems the only reliable solution. For whatever reason,
    // setXORMode is not working properly.
    if (drag != null) {
      g.setColor(Params._t_selectedcolor);
      g.drawLine(firstState.p.x, firstState.p.y, drag.x, drag.y);
    }

    for (Enumeration e = theStates.elements(); e.hasMoreElements();) {
      ((State) e.nextElement()).paint(g, showDfaLabels);
    }

    if (initialState != null) {
      setTransitions(initialState, false);
      for(Enumeration e = theTransitions.elements(); e.hasMoreElements();) {
	aTransition = (Transition) e.nextElement();
	if(aTransition.to == initialState) {
	  initialState.transitions.addElement(
	     new Transition(aTransition.to, aTransition.from, 1, new String[2]));
	}
      }
      initialState.paintInitial(g, this);
      setTransitions(initialState, true);
    }

    g.setFont(Params._t_font);
    for (Enumeration e = theTransitions.elements(); e.hasMoreElements();) {
      aTransition = (Transition) e.nextElement();
      aTransition.paintLabel(g, selectedTransition, locateLabel(aTransition));
    }

    if(showDfaLabels) {
      for (Enumeration e = theStates.elements(); e.hasMoreElements();) {
	((State) e.nextElement()).paintDfaLabel(g, selectedState);
      }
    }

    if (selectedTransition != null)
      if (theTransitions.indexOf(selectedTransition) != -1)
        selectedTransition.paintLabel(g, selectedTransition, locateLabel(selectedTransition));

    if (menuOn)
      menu.paint(g);

  }	// end of method paint.


  public void update(Graphics g) {

    Dimension	d = getSize();

    if (offscr == null) {
      offscr = createImage(d.width, d.height);
      offg = offscr.getGraphics();
    }
    if (offscr.getWidth(this) != d.width || offscr.getHeight(this) != d.height) {
      offscr = createImage(d.width, d.height);
      offg = offscr.getGraphics();
    }

    if (offg == null) 
      paint(g);
    else {
      paint(offg);
      g.drawImage(offscr, 0, 0, null);
    }
  }	// end of method update.

  /**
   * Creates a new state and adds it to the list of states.
   */
  public State addState(int x, int y) {
    State s = new State(currentId++, false, x, y);
    theStates.addElement(s);
    return s;
  }

  /**
    * returns the State with the given id if there is one
    * else returns null.
    */
  public State getStateWithId(int idNum) {
    State aState;
    for(Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
      aState = (State) e.nextElement();
      if(aState.id == idNum)
	return aState;
    }
    return null;
  }

  /**
   * Checks to see if the given position is in a state.
   * To speed it up, we check if the position is in the
   * rectangle surrounding the state's circle.
   * @return	the state found, or null, if none was found.
   */
  public State isInState(int x, int y) {
    State	s = null;
    boolean	found = false;
    Enumeration	e = theStates.elements();

    while ((e.hasMoreElements()) && (!found)) {
      s = (State) e.nextElement();
      if ((Math.abs(s.p.x - x) < Params._s_radius) && (Math.abs(s.p.y - y) < Params._s_radius))
        found = true;
    }
    if (found)
      return s;
    else
      return null;
  }	// end of method isInState.

  /**
   * Creates a new transition and adds it to the list of transitions.
   */
  public Transition addTransition(State from, State to) {
    String label = "";
    
    Transition t = new Transition(from, to, label, fieldCounter, separator);
    theTransitions.addElement(t);
    return t;
  }

  /**
   * Removes the given transition from the list of transitions.
   */
  public void removeTransition(Transition t) {
    if (selectedTransition == t)
      selectedTransition = null;
    theTransitions.removeElement(t);
  }

  /** 
    * Removes the given State from the Desktop
    */
  public void removeState(State s) {
    Transition tt;
    int i = 0;
    while (i < theTransitions.size()) {
      tt = (Transition) theTransitions.elementAt(i);
      if (tt.from == s || tt.to == s) 
	removeTransition(tt);
      else
	i++;
    }
    if (s == initialState)
      initialState = null;
    if(s == selectedState)
      selectedState = null;
    
    theStates.removeElement(s);
    
    if (autoRelabel)
      relabel();
  }
  
  /**
   * Returns the position of the label of the given transition.
   */
  public Point locateLabel(Transition t) {
    int idx = theTransitions.indexOf(t);
    FontMetrics	fm = getFontMetrics(Params._t_font);
    Transition tt;
    int x1 = t.from.p.x;
    int y1 = t.from.p.y;
    int x2 = t.to.p.x;
    int y2 = t.to.p.y;
    int x, y;

    count = 1;

    for (int i = 0; i < idx; i++) {
      tt = (Transition) theTransitions.elementAt(i);
      if (tt.from == t.from && tt.to == t.to)
        count++;
    }

    if (t.from == t.to) {
      y = (int) (y1 - t.distance + count*Params._t_height*df);
      y = Math.min(y, y1 - Params._t_distance);
      x = x1;
    } else {
      double len = Math.sqrt((x2-x1)*(x2-x1) + (y2-y1)*(y2-y1));
      double d = (count*(Params._t_height*df));
      d = Math.min(d, (len/2 - Params._t_height/2 + 1));
      d = (len == 0) ? 0 : d/len;
      x = (int) (x1 - ((x1 - x2)*d));
      y = (int) (y1 - ((y1 - y2)*d));
      
      if (x1 > x2){
	x -=  fm.stringWidth(t.label)+Transition.off+Transition.offr;
	if(selectedTransition == t)
	  x -= fm.stringWidth("_");
      }     
      if(y1 > y2) {
	y -= Params._t_height;

      }
    }

    return(new Point(x, y));
  }	// end of method locateLabel.

  /**
   * Checks to see if the given position is in a label.
   * @return	the transition found, or null if none was found.
   */
  public Transition isInTransition(int x, int y) {
    Transition	t = null;
    boolean	found = false;
    int		tx, ty;
    Point 	pl;
    FontMetrics	fm = getFontMetrics(Params._t_font);
    Vector	rt = new Vector(theTransitions.size());	// will keep trans. in reversed order.
    for (Enumeration e = theTransitions.elements(); e.hasMoreElements(); ) 
      rt.insertElementAt(e.nextElement(), 0);  
    Enumeration	e = rt.elements();
    while ((e.hasMoreElements()) && (!found)) {
      t = (Transition) e.nextElement();
      pl = locateLabel(t);
      tx = pl.x;
      ty = pl.y;
      if (((x - tx) <= fm.stringWidth(t.label)+Transition.off+Transition.offr) && ((x - tx) >= 0) && ((y - ty) <= Params._t_height) && ((y - ty) >= 0))
        found = true;
    }
    if (found)
      return t;
    else
      return null;
  }	// end of method isInTransition.


  /**
    * Checks if the given position is in a Dfa State label.
    * @return the corresponding State found, or null if none is found
    */
  public State isInStateDfaLabel(int x , int y) {
    State s = null;
   
    for(Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
      s = (State) e.nextElement();
      if((x >= s.labelStart.x) && (x <= s.labelEnd.x) && 
	 (y >= s.labelStart.y) && (y <= s.labelEnd.y))
	return s;
    }
    return null;
  }

  /**
   * Checks to see if the given position is a good
   * location for a new state. Not implemented.
   */
  boolean isGoodLocation(int x, int y) {
    return true;
  }	// end of method isGoodLocation.

  /**
   * Changes the aspect of the desktop.
   * It becomes faded when fade is true, and unfaded when fade is false.
   */
  public void changeAspect(boolean fade) {
    State s;

    if (fade) {			// upper window on; faded desktop
      selectedTransition = null;
      backcolor = Params._d_fadedbackcolor;
    } else {			// upper window closed; sharp desktop
      backcolor = Params._d_backcolor;
      for (Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
        s = (State) e.nextElement();
        s.ss = State.NORMAL;
      }
    } 
    repaint();
  }  // end of method changeAspect
  
  /**
   * Relabels the states to have consecutive labels.
   */
  public void relabel() {
    int newid = 0;
    State s = null;
    for (Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
      s = (State) e.nextElement();
      s.id = newid++;
    }
    currentId = newid;
  }	// end of method relabel.
   
  /**
   * Removes the highlights from all the states.
   */
  public void unfocusAll() {
    State	s = null;

    for (Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
      s = (State) e.nextElement();
      s.ss = State.NORMAL;
    }
  }


  public void setAllStateLists(Desktop d) {
    for (Enumeration e = theStates.elements(); e.hasMoreElements(); ) 
      ((State) e.nextElement()).labelToStateList(d);
  }


  /**
    * Unselects all transitions on the Desktop
    */
 public void unselectTransitions() {
    Transition aTransition;
    selectedTransition = null;
    for(Enumeration e = theTransitions.elements(); e.hasMoreElements(); ) {
      aTransition = (Transition) e.nextElement();
      aTransition.st = Transition.NORMAL;
    }
  }

  /**
   * Returns a string representing the tabular form of the current machine.
   * It is the only method we could find to print the machine on all platforms.
   */
  public String toTabular() {
  
    int		cols = theStates.size() + 1;
    int		rows = theTransitions.size() + theStates.size();
    String[][]	table = new String[rows][cols];
    String[]	header = new String[cols];
    int[]	colWidth = new int[cols];
    int		currentCol = 0;
    int		currentRow = 0;
    int		rowSpan = 0;
    State	s;
    Transition	t;
    String	answer = "";
    String	finalstates = "Final States: ";
    String	initialstate = "Start state: " 
    			+ ((initialState != null) ? "q" + initialState.id : "");
    if (theStates.size() == 0)
      return " ";
    
    for (int i = 0; i < cols; i++) {
      for (int j = 0; j < rows; j++)
        table[j][i] = EMPTY_STR;
      colWidth[i] = 0;
    }
    
    for (Enumeration e1 = theStates.elements(); e1.hasMoreElements(); ) {
      s = (State) e1.nextElement();
      if (s.isFinal)
        finalstates += " q" + s.id;
      table[currentRow][0] = "q" + s.id + "\t|";
      rowSpan = 1;
      setTransitions(s, false);
      for (Enumeration e2 = s.transitions.elements(); e2.hasMoreElements(); ) {
        t = (Transition) e2.nextElement();
        currentCol = theStates.indexOf(t.to) + 1;
        if (t.label.length() > colWidth[currentCol])
          colWidth[currentCol] = t.label.length();
        int i = 0;
        while (!table[currentRow + i][currentCol].equals(EMPTY_STR)) {
          i++;
          table[currentRow + i][0] = "\t|";
        }
        table[currentRow + i][currentCol] = t.label.trim();
        if (i + 1 > rowSpan)
          rowSpan = i + 1;
      }
      currentRow += rowSpan;
    }
    rows = currentRow;
    
    String str;
    int totalWidth = 0;
    answer = "";
    answer += initialstate + "\n";
    answer += finalstates + "\n";
    answer += "Transition function: " + "\n";
    answer += "\n\t ";
    for (Enumeration e1 = theStates.elements(); e1.hasMoreElements(); ) {
      s = (State) e1.nextElement();
      str = "q" + s.id;
      currentCol = theStates.indexOf(s) + 1;
      header[currentCol] = " " + toLength(str, colWidth[currentCol])+ " ";
      answer += header[currentCol];
      totalWidth += Math.max(colWidth[currentCol], 3) + 2;
    }
    answer += "\n";
    answer += "\t-";
    for (int i = 0; i < totalWidth; i++) 
      answer += "-";
    answer += "\n";
    for (int i = 0; i < rows; i++) {
      if (!table[i][0].equals("\t|") && (i > 0)) {
        answer += "\t|";
        for (int ii = 0; ii < totalWidth; ii++) 
          answer += "-";
        answer += "\n";
      }
      answer += " " + table[i][0];
      for (int j = 1; j < cols; j++) {
        table[i][j] = " " + toLength(table[i][j], colWidth[j]) + "|";
        answer += table[i][j];
      }
      answer += "\n";
    }
    answer += "\t-";
    for (int i = 0; i < totalWidth; i++) 
      answer += "-";
    answer += "\n";
    return answer;
  }	// end of method toTabular.
  
  /**
   * Makes the String str have length length by adding white spaces.
   * Used in method toTabular.
   */
  protected String toLength(String str, int length) {
    String	answer = new String(str);
    int		actuall = Math.max(3, length);
    
    if (answer.equals(EMPTY_STR))
      answer = "-";
    int l = answer.length();
    for (int i = l; i < actuall; i++) 
      answer += " ";
    return answer;
  }

  /** sets the checkMarks on all the States on this Desktop to false  */
  public void unmarkAllStates() {
    State aState;

    for( Enumeration e = theStates.elements(); e.hasMoreElements(); ) {
      aState = (State) e.nextElement();
      aState.checkMark = 0;
    }
  }

  //*************** functions for placing the nodes when solving the DFA *************************

  /** calculates how many Transitions intersect transitions or states on the Desktop */  
  private int findIntersections(State aState, Vector placedStates) {
    int total = 0;
    Enumeration e, e2, e3;
    Transition aTransition, aTransition2;
    Point [] points = new Point[5];
    State aState2;
    int i;
    Point p1, p2;

    placedStates.addElement(aState);
    
    for(e = theTransitions.elements(); e.hasMoreElements(); ) {
      aTransition = (Transition) e.nextElement();
      if(aTransition.from == aTransition.to) {
	p1 = new Point(aTransition.from.p);
	p1.y -= aTransition.distance; 
      }
      else p1 = new Point(aTransition.to.p);
      
      if(placedStates.contains(aTransition.to) && placedStates.contains(aTransition.from)) {
	for(e2 = theTransitions.elements(); e2.hasMoreElements(); ) {
	  aTransition2 = (Transition) e2.nextElement();
	  if(placedStates.contains(aTransition2.to) && placedStates.contains(aTransition.from)) {
	    
	    if(aTransition2.from == aTransition2.to) {
	      p2 = new Point(aTransition2.from.p);
	      p2.y -= aTransition2.distance; 
	    }
	    else p2 = new Point(aTransition2.to.p);
	    
	    // same state transitions going off the top of the screen are bad
	    if(p1.y < 0 || p2.y <0)
	      total++;

	    if(intersect(aTransition.from.p, p1, aTransition2.from.p, p2))
	       total++;
	  }
	}
	for( e3 = placedStates.elements(); e3.hasMoreElements(); ) {
	  aState2 = (State) e3.nextElement();
	  points[0] = new Point(aState2.p);
	  points[0].x -= Params._s_radius;
	  points[0].y -=  Params._s_radius;
	  points[1] = new Point(points[0]);
	  points[1].x += 2* Params._s_radius;
	  points[2] = new Point(points[0]);
	  points[2].y += 2* Params._s_radius;
	  points[3] = new Point(points[2]);
	  points[3].x += 2*  Params._s_radius;
	  points[4] = new Point(points[0]);
	  for(i=0; i<4; i++) {
	    if(intersect(aTransition.from.p, p1, points[i], points[i+1])) {
	      total+=2;
	      break;
	    }
	  }
	}
      }
    }
    placedStates.removeElement(aState);
    return total;
  }


  /** cool function from Sedgewicks book to figure things about a line's orientation. 
    * Useful for Intersect.*/
  private int ccw(Point p0, Point p1, Point p2) {
    int dx1, dx2, dy1, dy2;
    int ccw = 0;
    dx1 = p1.x - p0.x;
    dy1 = p1.y - p0.y;
    dx2 = p2.x - p0.x;
    dy2 = p2.y - p0.y;

    if(dx1*dy2 > dy1 * dx2) 
      ccw = 1;
    if(dx1*dy2 < dy1 * dx2)
      ccw = -1;
    if(dx1*dy2 == dy1*dx2) {
      if((dx1*dx2 < 0) || (dy1*dy2<0))
	ccw = -1;
      else if ((dx1*dx1+dy1+dy1) >= (dx2*dx2+dy2*dy2))
	ccw = 0;
      else
	ccw = 1;
    }
    return ccw;
  }

  /** checks if two line segments intersect */
  private boolean intersect(Point p1, Point p2, Point q1, Point q2) {
    return ((ccw(p1, p2, q1) * ccw(p1, p2, q2)) <= 0) && 
      ((ccw(q1, q2, p1)*ccw(q1, q2, p2)) <=0);
  }

  /** tries to find a good position for the parameter State by trying a few random positions
    * and keeping the best one.     */
  public void placeState(State aState, Vector placedStates) {
    Point bestPoint = new Point(-1, -1);
    int bestTotal = 1000;
    int total= 0;
    int counter = 0;
    Enumeration e;
    Transition aTransition;
    State neighbour;
    int distancex, distancey;
    boolean found = false;
    Dimension	dim = getSize();
    

    //give it ten tries and keep the best
    for (int i=0; i<10; i++) {
      for(e = theTransitions.elements(); e.hasMoreElements(); ) {
	aTransition = (Transition) e.nextElement();
	if(aTransition.from == aState && placedStates.contains(aTransition.to))
	  neighbour = aTransition.to;
	else if(aTransition.to == aState && placedStates.contains(aTransition.from))
	  neighbour = aTransition.from;
	else 
	  continue;
	if(counter != i) {
	  counter++;
	  continue; 
	}
	distancex = (int) ((Math.random() - .5) * 200.0);	
	distancey = (int) ((Math.random() - .5) * 160.0);
	
	if(distancex <= 0) 
	  distancex -= 2* Params._s_radius;
	else distancex += 2* Params._s_radius;
	if(distancey <= 0) 
	  distancey -= 2* Params._s_radius;
	else distancey += 2* Params._s_radius;	

	if(Math.random() > .50) {
	  distancex = 0;
	  distancey *= 2;
	} else if(Math.random() < .25) {
	  distancey = 0;
	  distancex *= 2;
	}
	aState.p.x = neighbour.p.x + distancex;
	aState.p.y = neighbour.p.y + distancey;
	findProperPosition(aState, placedStates);
	total = findIntersections(aState, placedStates);

	if(total < bestTotal) {
	  bestPoint = new Point(aState.p);
	  bestTotal = total;
	}
	found = true;
	break;
      }
      if(found == false) {
	// if all else fails go random
	aState.p.x = (int) (Math.random() * dim.width);	
	aState.p.y = (int) (Math.random()* dim.height);
	findProperPosition(aState, placedStates);
	total = findIntersections(aState, placedStates);

	if(total < bestTotal) {
	  bestPoint = new Point(aState.p);
	  bestTotal = total;
	}
      }
      found = false;
      counter = 0;
    }
    aState.p = new Point(bestPoint);
  }

  /** makes sure the State isn't overlapping another state or is placed off the screen */
  private void findProperPosition(State aState, Vector placedStates) {
    Point pos = aState.p;
    Dimension dim = getSize();
    State tempState;
    boolean okay = false;
    int count = 0;

    while (okay == false) {
      
      if(pos.x <= Params._s_radius)
	pos.x = 3 *  Params._s_radius;
      else if(pos.y <= Params._s_radius)
	pos.y = 3 *  Params._s_radius ;
      else if(pos.x >= dim.width - Params._s_radius)
	pos.x = dim.width - (2* Params._s_radius);
      else if(pos.y >= dim.height - Params._s_radius)
	pos.y = dim.height - (2* Params._s_radius);  
      else okay = true;
      
      if(count++ == 50)
	return;
  
      for(Enumeration e = placedStates.elements(); e.hasMoreElements(); ) {
	tempState = (State) e.nextElement();
	if(( Math.abs(tempState.p.y - aState.p.y) <= (Params._s_radius*2)) && 
	   ( Math.abs(tempState.p.x - aState.p.x) <= (Params._s_radius*2))) {
	  aState.p.x += 5* Params._s_radius;
	  aState.p.y += 5* Params._s_radius;
	  okay = false;
	}
      }
    }
  }



}

    
 


